% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_distance_info.R, R/tree_distance_msi.R
\encoding{UTF-8}
\name{TreeDistance}
\alias{TreeDistance}
\alias{SharedPhylogeneticInfo}
\alias{DifferentPhylogeneticInfo}
\alias{PhylogeneticInfoDistance}
\alias{ClusteringInfoDistance}
\alias{ClusteringInfoDist}
\alias{ExpectedVariation}
\alias{MutualClusteringInfo}
\alias{MutualClusteringInformation}
\alias{SharedPhylogeneticInfoSplits}
\alias{MutualClusteringInfoSplits}
\alias{MatchingSplitInfo}
\alias{MatchingSplitInfoDistance}
\alias{MatchingSplitInfoSplits}
\title{Information-based generalized Robinson–Foulds distances}
\usage{
TreeDistance(tree1, tree2 = NULL)

SharedPhylogeneticInfo(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE,
  diag = TRUE
)

DifferentPhylogeneticInfo(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE
)

PhylogeneticInfoDistance(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE
)

ClusteringInfoDistance(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE
)

ExpectedVariation(tree1, tree2, samples = 10000)

MutualClusteringInfo(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE,
  diag = TRUE
)

SharedPhylogeneticInfoSplits(
  splits1,
  splits2,
  nTip = attr(splits1, "nTip"),
  reportMatching = FALSE
)

MutualClusteringInfoSplits(
  splits1,
  splits2,
  nTip = attr(splits1, "nTip"),
  reportMatching = FALSE
)

MatchingSplitInfo(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE,
  diag = TRUE
)

MatchingSplitInfoDistance(
  tree1,
  tree2 = NULL,
  normalize = FALSE,
  reportMatching = FALSE
)

MatchingSplitInfoSplits(
  splits1,
  splits2,
  nTip = attr(splits1, "nTip"),
  reportMatching = FALSE
)
}
\arguments{
\item{tree1, tree2}{Trees of class \code{phylo}, with leaves labelled identically,
or lists of such trees to undergo pairwise comparison.  Where implemented,
\code{tree2 = NULL} will compute distances between each pair of trees in the list
\code{tree1} using a fast algorithm based on
\insertCite{Day1985;textual}{TreeDist}.}

\item{normalize}{If a numeric value is provided, this will be used as a
maximum value against which to rescale results.
If \code{TRUE}, results will be rescaled against a maximum value calculated from
the specified tree sizes and topology, as specified in the "Normalization"
section below.
If \code{FALSE}, results will not be rescaled.}

\item{reportMatching}{Logical specifying whether to return the clade
matchings as an attribute of the score.}

\item{diag}{Logical specifying whether to return similarities along the
diagonal, i.e. of each tree with itself.  Applies only if \code{tree2} is
a list identical to \code{tree1}, or \code{NULL}.}

\item{samples}{Integer specifying how many samplings to obtain;
accuracy of estimate increases with \code{sqrt(samples)}.}

\item{splits1, splits2}{Logical matrices where each row corresponds to a leaf,
either listed in the same order or bearing identical names (in any sequence),
and each column corresponds to a split, such that each leaf is identified as
a member of the ingroup (\code{TRUE}) or outgroup (\code{FALSE}) of the respective
split.}

\item{nTip}{(Optional) Integer specifying the number of leaves in each split.}
}
\value{
If \code{reportMatching = FALSE}, the functions return a numeric
vector specifying the requested similarities or differences.

If \code{reportMatching = TRUE}, the functions additionally return an integer
vector listing the index of the split in \code{tree2} that is matched with
each split in \code{tree1} in the optimal matching.
Unmatched splits are denoted \code{NA}.
Use \code{\link[=VisualizeMatching]{VisualizeMatching()}} to plot the optimal matching.

\code{TreeDistance()} simply returns the clustering information distance (it is
an alias of \code{ClusteringInfoDistance()}).
}
\description{
Calculate tree similarity and distance measures based on the amount of
phylogenetic or clustering information that two trees hold in common, as
proposed in Smith (2020).
}
\details{
\href{https://ms609.github.io/TreeDist/articles/Robinson-Foulds.html#generalized-robinson-foulds-distances}{Generalized Robinson–Foulds distances}
calculate tree similarity by finding an
optimal matching that the similarity between a split on one tree
and its pair on a second, considering all possible ways to pair splits
between trees (including leaving a split unpaired).

The methods implemented here use the concepts of
\href{https://ms609.github.io/TreeDist/articles/information.html}{entropy and information}
\insertCite{Mackay2003}{TreeDist} to assign a similarity score between each
pair of splits.

The returned tree similarity measures state the amount of information,
in bits, that the splits in two trees hold in common
when they are optimally matched, following
\insertCite{SmithDist;textual}{TreeDist}.
The complementary tree distance measures state how much information is
different in the splits of two trees, under an optimal matching.
Where trees contain different tips, tips present in one tree but not the
other are removed before each comparison (as by definition, the trees neither
hold information in common nor differ regarding these tips).
}
\section{Concepts of information}{
The phylogenetic (Shannon) information content and entropy of a split are
defined in
\href{https://ms609.github.io/TreeDist/articles/information.html}{a separate vignette}.

Using the mutual (clustering) information
\insertCite{Meila2007,Vinh2010}{TreeDist} of two splits to quantify their
similarity gives rise to the Mutual Clustering Information measure
(\code{MutualClusteringInfo()}, \code{MutualClusteringInfoSplits()});
the entropy distance gives the Clustering Information Distance
(\code{ClusteringInfoDistance()}).
This approach is optimal in many regards, and is implemented with
normalization in the convenience function \code{TreeDistance()}.

Using the amount of phylogenetic information common to two splits to measure
their similarity gives rise to the Shared Phylogenetic Information similarity
measure (\code{SharedPhylogeneticInfo()}, \code{SharedPhylogeneticInfoSplits()}).
The amount of information distinct to
each of a pair of splits provides the complementary Different Phylogenetic
Information distance metric (\code{DifferentPhylogeneticInfo()}).

The Matching Split Information measure (\code{MatchingSplitInfo()},
\code{MatchingSplitInfoSplits()}) defines the similarity between a pair of
splits as the phylogenetic information content of the most informative
split that is consistent with both input splits; \code{MatchingSplitInfoDistance()}
is the corresponding measure of tree difference.
(\href{https://ms609.github.io/TreeDist/articles/Generalized-RF.html}{More information here}.)
}

\section{Conversion to distances}{
To convert similarity measures to distances, it is necessary to
subtract the similarity score from a maximum value.  In order to generate
distance \emph{metrics}, these functions subtract the similarity twice from the
total information content (SPI, MSI) or entropy (MCI) of all the splits in
both trees \insertCite{SmithDist}{TreeDist}.
}

\section{Normalization}{
If \code{normalize = TRUE}, then results will be rescaled such that distance
ranges from zero to (in principle) one.
The maximum \strong{distance} is the sum of the information content or entropy of
each split in each tree; the maximum \strong{similarity} is half this value.
(See Vinh \emph{et al.} (2010, table 3) and
\insertCite{SmithDist;textual}{TreeDist} for
alternative normalization possibilities.)

Note that a distance value of one (= similarity of zero) will seldom be
achieved, as even the most different trees exhibit some similarity.
It may thus be helpful to rescale the normalized value such that the
\emph{expected} distance between a random pair of trees equals one.  This can
be calculated with \code{ExpectedVariation()}; or see package
'\href{https://ms609.github.io/TreeDistData/reference/randomTreeDistances.html}{TreeDistData}'
for a compilation of expected values under different metrics for trees with
up to 200 leaves.

Alternatively, use \verb{normalize = }\code{\link{pmax}} or \code{\link{pmin}} to scale against the
information content or entropy of all splits in the most (\code{pmax}) or
least (\code{pmin}) informative tree in each pair.
To calculate the relative similarity against a reference tree that is known
to be "correct", use \code{normalize = SplitwiseInfo(trueTree)} (SPI, MSI) or
\code{ClusteringEntropy(trueTree)} (MCI).
For worked examples, see the internal function \code{\link[=NormalizeInfo]{NormalizeInfo()}}, which is
called from distance functions with the parameter \code{how = normalize}.
.
}

\section{Distances between large trees}{
To balance memory demands and runtime with flexibility, these functions are
implemented for trees with up to 2048 leaves.
To analyse trees with up to 8192 leaves, you will need to a modified version
of the package:
\code{install.packages("BigTreeDist", repos = "https://ms609.github.io/packages/")}
Use \code{library("BigTreeDist")} \emph{instead} of \code{library("TreeDist")} to load
the modified package – or prefix functions with the package name, e.g.
\code{BigTreeDist::TreeDistance()}.

As an alternative download method,
uninstall \pkg{TreeDist} and \pkg{TreeTools} using
\code{remove.packages()}, then use
\code{devtools::install_github("ms609/TreeTools", ref = "more-leaves")}
to install the modified \pkg{TreeTools} package; then,
install \pkg{TreeDist} using
\code{devtools::install_github("ms609/TreeDist", ref = "more-leaves")}.
(\pkg{TreeDist} will need building from source \emph{after} the modified
\pkg{TreeTools} package has been installed, as its code links to values
set in the TreeTools source code.)

Trees with over 8192 leaves require further modification of the source code,
which the maintainer plans to attempt in the future; please \href{https://github.com/ms609/TreeTools/issues/141}{comment on GitHub} if you would find this useful.
}

\examples{
tree1 <- ape::read.tree(text="((((a, b), c), d), (e, (f, (g, h))));")
tree2 <- ape::read.tree(text="(((a, b), (c, d)), ((e, f), (g, h)));")
tree3 <- ape::read.tree(text="((((h, b), c), d), (e, (f, (g, a))));")

# Best possible score is obtained by matching a tree with itself
DifferentPhylogeneticInfo(tree1, tree1) # 0, by definition
SharedPhylogeneticInfo(tree1, tree1)
SplitwiseInfo(tree1) # Maximum shared phylogenetic information

# Best possible score is a function of tree shape; the splits within
# balanced trees are more independent and thus contain less information
SplitwiseInfo(tree2)

# How similar are two trees?
SharedPhylogeneticInfo(tree1, tree2) # Amount of phylogenetic information in common
attr(SharedPhylogeneticInfo(tree1, tree2, reportMatching = TRUE), "matching")
VisualizeMatching(SharedPhylogeneticInfo, tree1, tree2) # Which clades are matched?

DifferentPhylogeneticInfo(tree1, tree2) # Distance measure
DifferentPhylogeneticInfo(tree2, tree1) # The metric is symmetric

# Are they more similar than two trees of this shape would be by chance?
ExpectedVariation(tree1, tree2, sample=12)["DifferentPhylogeneticInfo", "Estimate"]

# Every split in tree1 conflicts with every split in tree3
# Pairs of conflicting splits contain clustering, but not phylogenetic, 
# information
SharedPhylogeneticInfo(tree1, tree3) # = 0
MutualClusteringInfo(tree1, tree3) # > 0

# Distance functions internally convert trees to Splits objects.
# Pre-conversion can reduce run time if the same trees will feature in
# multiple comparisons
splits1 <- TreeTools::as.Splits(tree1)
splits2 <- TreeTools::as.Splits(tree2)

SharedPhylogeneticInfoSplits(splits1, splits2)
MatchingSplitInfoSplits(splits1, splits2)
MutualClusteringInfoSplits(splits1, splits2)
}
\references{
\insertAllCited{}
}
\seealso{
Other tree distances: 
\code{\link{JaccardRobinsonFoulds}()},
\code{\link{KendallColijn}()},
\code{\link{MASTSize}()},
\code{\link{MatchingSplitDistance}()},
\code{\link{NNIDist}()},
\code{\link{NyeSimilarity}()},
\code{\link{PathDist}()},
\code{\link{Robinson-Foulds}},
\code{\link{SPRDist}()}
}
\author{
\href{https://orcid.org/0000-0001-5660-1727}{Martin R. Smith}
(\href{mailto:martin.smith@durham.ac.uk}{martin.smith@durham.ac.uk})
}
\concept{tree distances}
