% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcStewart.R
\name{mcStewart}
\alias{mcStewart}
\title{Stewart Potentials Parallel}
\usage{
mcStewart(
  knownpts,
  unknownpts,
  varname,
  typefct = "exponential",
  span,
  beta,
  resolution,
  mask,
  cl,
  size = 1000,
  longlat = TRUE,
  returnclass = "sp"
)
}
\arguments{
\item{knownpts}{sp or sf object; this is the set of known observations to 
estimate the potentials from.}

\item{unknownpts}{sp or sf object; this is the set of unknown units for which 
the function computes the estimates. Not used when \code{resolution} is set 
up. (optional)}

\item{varname}{character; name of the variable in the \code{knownpts} dataframe 
from which potentials are computed. Quantitative variable with no negative values.}

\item{typefct}{character; spatial interaction function. Options are "pareto" 
(means power law) or "exponential".
If "pareto" the interaction is defined as: (1 + alpha * mDistance) ^ (-beta).
If "exponential" the interaction is defined as: 
exp(- alpha * mDistance ^ beta).
The alpha parameter is computed from parameters given by the user 
(\code{beta} and \code{span}).}

\item{span}{numeric; distance where the density of probability of the spatial 
interaction function equals 0.5.}

\item{beta}{numeric; impedance factor for the spatial interaction function.}

\item{resolution}{numeric; resolution of the output SpatialPointsDataFrame
(in map units). If resolution is not set, the grid will contain around 7250 
points. (optional)}

\item{mask}{sp or sf object; the spatial extent of this object is used to 
create the regularly spaced points output. (optional)}

\item{cl}{numeric; number of clusters. By default cl is determined using 
\code{parallel::detectCores()}.}

\item{size}{numeric; mcStewart splits unknownpts in chunks, size indicates 
the size of each chunks.}

\item{longlat}{logical; if FALSE, Euclidean distance, if TRUE Great Circle 
(WGS84 ellipsoid) distance.}

\item{returnclass}{"sp" or "sf"; class of the returned object.}
}
\value{
Point object with the computed potentials in a new field 
named \code{OUTPUT}.
}
\description{
This function computes Stewart potentials using parallel 
computation.
}
\details{
The parallel implementation splits potentials computations along 
chunks of unknownpts (or chunks of the grid defined using resolution).
}
\examples{
\dontrun{
if(require(cartography)){
  nuts3.spdf@data <- nuts3.df
  t1 <- system.time(
    s1 <- stewart(knownpts = nuts3.spdf,resolution = 40000,
                  varname = "pop2008",
                  typefct = "exponential", span = 100000,
                  beta = 3, mask = nuts3.spdf, returnclass = "sf")
  )
  t2 <- system.time(
    s2 <- mcStewart(knownpts = nuts3.spdf, resolution = 40000,
                    varname = "pop2008",
                    typefct = "exponential", span = 100000,
                    beta = 3, mask = nuts3.spdf, cl = 3, size = 500, 
                    returnclass = "sf")
  )
  identical(s1, s2)
  cat("Elapsed time\n", "stewart:", t1[3], "\n mcStewart:",t2[3])
  
  iso <- isopoly(x = s2, 
                 breaks = c(0,1000000,2000000, 5000000, 10000000, 20000000, 
                            200004342),
                 mask = nuts3.spdf, returnclass = "sf")
  # cartography
  opar <- par(mar = c(0,0,1.2,0))
  bks <- sort(unique(c(iso$min, iso$max)))
  choroLayer(x = iso, var = "center", breaks = bks, border = NA,
             legend.title.txt = "pop")
  layoutLayer("potential population", "","", scale = NULL)
  par(opar)
}
}
}
\seealso{
\link{stewart}.
}
