\name{condmap}
\alias{condmap}
\title{Produces a conditional 2D map from a fitted max-stable process}
\description{
  Produces a conditional 2D map from a fitted max-stable process.
}
\usage{
condmap(fitted, fix.coord, x, y, marg.cov = NULL, \dots, ret.per1 = 100,
ret.per2 = ret.per1, col = terrain.colors(length(x)), plot.contour =
TRUE)
}
\arguments{
  \item{fitted}{An object of class \code{maxstab}. Most often, it will
    be the output of the function \code{\link{fitmaxstab}}.}
  \item{fix.coord}{The spatial coordinates of the location from which
    the conditional quantile is computed.}
  \item{x,y}{Numeric vector defining the grid at which the levels are
    computed.}
  \item{marg.cov}{A matrix of the (currently only one) marginal
    covariable value at points x[i], y[j]. See function
    \code{\link{fitmaxstab}}.}
  \item{\dots}{Several arguments to be passed to the \code{link{image}}
    and \code{\link{contour}} functions.}
  \item{ret.per1,ret.per2}{Numerics giving the return period for which
    the quantile map is plotted. See details.}
  \item{col}{A list of colors such as that generated by 'rainbow',
    'heat.colors', 'topo.colors', 'terrain.colors' or similar
    functions.}
  \item{plot.contour}{Logical. If \code{TRUE} (default), contour lines
    are added to the plot.}
}
\value{
  A plot. Additionally, a list with the details for plotting the map is
  returned invisibly.
}
\details{
  The function solves the following equation:

  \deqn{\Pr\left[Z(x_2) > z_2 | Z(x_1) > z_1 \right] =
    \frac{1}{ret.per2}}{Pr[Z(x_2) > z_2 | Z(x_1) > z_1] = 1 / ret.per2}
  where \eqn{z_1 = -1 / \log(1 - 1/ret.per1)}{z_1 = -1 / \log(1 -
    1/ret.per1)}.

  In other words, it computes, given that at location \eqn{x_1}{x_1} we
  exceed the level \eqn{z_1}{z_1}, the levels which is expected to be
  exceeded in average every \code{ret.per2} year.
}
\seealso{
  \code{\link{map}}, \code{\link{filled.contour}},
  \code{\link{heatmap}}, \code{\link{heat.colors}},
  \code{\link{topo.colors}}, \code{\link{terrain.colors}},
  \code{\link{rainbow}}
}
\examples{
\dontrun{
require(RandomFields)

##Define the coordinate of each location
n.site <- 30
locations <- matrix(runif(2*n.site, 0, 10), ncol = 2)
colnames(locations) <- c("lon", "lat")

sigma <- matrix(c(4, 1, 1, 3), 2)
sigma.inv <- solve(sigma)
sqrtCinv <- t(chol(sigma.inv))
model <- list(list(model = "gauss", var = 1, aniso = sqrtCinv / 2))

##Simulate a max-stable process - with unit Frechet margins
ms0 <- MaxStableRF(locations[,1], locations[,2], grid=FALSE,
                   model = model, maxstable = "Bool", n = 40)
ms0 <- t(ms0)
ms1 <- ms0

##Now define the spatial model for the GEV parameters
param.loc <- -10 - 4 * locations[,1] + locations[,2]^2
param.scale <- 5 + locations[,1] + locations[,2]^2 / 10
param.shape <- rep(.2, n.site)

##Transform the unit Frechet margins to GEV 
for (i in 1:n.site)
  ms1[,i] <- param.scale[i] * (ms1[,i]^param.shape[i] - 1) /
  param.shape[i] + param.loc[i]

##Define a model for the GEV margins to be fitted
##shape ~ 1 stands for the GEV shape parameter is constant
##over the region
loc.form <- loc ~ lon + I(lat^2)
scale.form <- scale ~ lon + I(lat^2)
shape.form <- shape ~ lat + lon

##  1- Fit a max-stable process
fitted <- fitmaxstab(ms1, locations, "gauss", loc.form, scale.form,
                     shape.form, std.err.type = "none")

condmap(fitted, c(1, 1), seq(0, 10, length = 25), seq(0,10, length =
25))
}
}
\author{Mathieu Ribatet}
\keyword{hplot}
