## SimInf, a framework for stochastic disease spread simulations
## Copyright (C) 2015 - 2017  Stefan Engblom
## Copyright (C) 2015 - 2017  Stefan Widgren
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

##' Create a DESCRIPTION file for the package skeleton
##' @noRd
create_DESCRIPTION_file <- function(path, name, author, maintainer,
                                    email, license)
{
    lines <- c(paste0("Package: ", name),
               "Type: Package",
               paste0("Title: Model ('", name, "') Interface to the 'SimInf' Package"),
               "Version: 1.0",
               paste0("Author: ", author),
               paste0("Maintainer: ", maintainer, " <", email, ">"),
               "Description: Generated by 'SimInf'. Provides an interface",
               paste0("    to the 'SimInf' package for the '", name, "' model."),
               paste0("License: ", license),
               "NeedsCompilation: yes",
               "Depends: SimInf",
               "Imports: methods",
               "LinkingTo: SimInf",
               "Collate:",
               "    'model.R'")

    writeLines(lines, con = file.path(path, "DESCRIPTION"))

    invisible(NULL)
}

##' Create a NAMESPACE file for the package skeleton
##' @noRd
create_NAMESPACE_file <- function(path, name)
{
    lines <- c(paste0("export(", name, ")"),
               paste0("exportClasses(", name, ")"),
               "exportMethods(run)",
               "import(SimInf)",
               "import(methods)",
               paste0("useDynLib(", name, ", .registration=TRUE)"))

    writeLines(lines, con = file.path(path, "NAMESPACE"))

    invisible(NULL)
}

create_model_C_file <- function(path, model, name)
{
    writeLines(C_code(model, name), con = file.path(path, "src", "model.c"))
    invisible(NULL)
}

create_model_R_class <- function(name)
{
    c(paste0("##' Class \\code{", name, "}"),
      "##'",
      paste0("##' Class to handle the \\code{", name, "} \\code{SimInf_model}."),
      "##' @export",
      paste0("setClass(\"", name, "\", contains = \"SimInf_model\")"))
}

create_model_R_object <- function(model, name)
{
    rows <- paste0(rownames(model@S), collapse = "\", \"")

    G <- utils::capture.output(dput(model@G))
    G <- c(paste0("G <- ", G[1]), G[-1])
    G <- paste0("    ", G)

    S <- utils::capture.output(dput(model@S))
    S <- c(paste0("S <- ", S[1]), S[-1])
    S <- paste0("    ", S)

    c("##' Create a model for the SimInf framework",
      "##'",
      "##' Create a model to be used by the SimInf framework.",
      "##' @param u0 A data.frame with the initial state in each node.",
      "##' @param tspan A vector (length >= 2) of increasing time points",
      "##'     where the state of each node is to be returned.",
      "##' @import SimInf",
      "##' @import methods",
      "##' @export",
      "##' @examples",
      "##' ## Please add example(s) how to use the model",
      paste0(name, " <- function(u0, tspan) {"),
      paste0("    compartments <- c(\"", rows, "\")"),
      "",
      "    ## Check u0",
      "    if (!is.data.frame(u0))",
      "        u0 <- as.data.frame(u0)",
      "    if (!all(compartments %in% names(u0)))",
      "        stop(\"Missing columns in u0\")",
      "    u0 <- u0[, compartments, drop = FALSE]",
      "",
      G,
      "",
      S,
      "",
      "    E <- methods::as(matrix(integer(0), nrow = 0, ncol = 0), \"dgCMatrix\")",
      "",
      "    N <- matrix(integer(0), nrow = 0, ncol = 0)",
      "",
      "    v0 <- matrix(numeric(0), nrow  = 0, ncol = nrow(u0))",
      "    storage.mode(v0) <- \"double\"",
      "",
      "    ldata <- matrix(numeric(0), nrow = 0, ncol = nrow(u0))",
      "    storage.mode(ldata) <- \"double\"",
      "",
      "    events <- NULL",
      "    gdata <- numeric()",
      "",
      "    model <- SimInf_model(G      = G,",
      "                          S      = S,",
      "                          E      = E,",
      "                          N      = N,",
      "                          tspan  = tspan,",
      "                          events = events,",
      "                          ldata  = ldata,",
      "                          gdata  = gdata,",
      "                          u0     = u0,",
      "                          v0     = v0)",
      "",
      paste0("    methods::as(model, \"", name, "\")"),
      "}")
}

create_model_run_fn <- function(name)
{
    c("##' Run the model",
      "##'",
      "##' @rdname run-methods",
      "##' @param model The model to run.",
      "##' @param threads Number of threads. Default is NULL, i.e. to use all",
      "##'     available processors.",
      "##' @param seed Random number seed. Default is NULL, i.e. the",
      "##'     simulator uses time to seed the random number generator.",
      "##' @return model with result from simulation.",
      "##' @export",
      paste0("##' @useDynLib ", name, ", .registration=TRUE"),
      "setMethod(\"run\",",
      paste0("    signature(model = \"", name, "\"),"),
      "    function(model, threads, seed)",
      "    {",
      "        methods::validObject(model)",
      paste0("       .Call(SimInf_model_run, model, threads, seed, PACKAGE = \"",
             name, "\")"),
      "    })")
}

create_model_R_file <- function(path, model, name)
{
    lines <- c(sprintf("## Generated by SimInf (v%s) %s */",
                       utils::packageVersion("SimInf"),
                       format(Sys.time(), "%Y-%m-%d %H:%M")),
               "",
               create_model_R_class(name),
               "",
               create_model_R_object(model, name),
               "",
               create_model_run_fn(name))

    writeLines(lines, con = file.path(path, "R", "model.R"))

    invisible(NULL)
}

create_model_class_man_file <- function(path, name)
{
    lines <- c("\\docType{class}",
               paste0("\\name{", name, "-class}"),
               paste0("\\alias{", name, "-class}"),
               paste0("\\title{Class \\code{\"", name, "\"}}"),
               "\\description{",
               paste0("Class to handle the ", name, " \\code{SimInf_model}."),
               "}")

    writeLines(lines, con = file.path(path, "man", paste0(name, "-class.Rd")))

    invisible(NULL)
}

create_model_man_file <- function(path, name)
{
    lines <- c(paste0("\\name{", name, "}"),
               paste0("\\alias{", name, "}"),
               "\\title{Create a model for the SimInf framework}",
               "\\usage{",
               paste0(name, "(u0, tspan)"),
               "}",
               "\\arguments{",
               "\\item{u0}{A \\code{data.frame} with the initial state in each node.}",
               "",
               "\\item{tspan}{A vector (length >= 2) of increasing time points",
               "where the state of each node is to be returned.}",
               "}",
               "\\description{",
               "Create a model to be used by the SimInf framework.",
               "}",
               "\\examples{",
               "## Please add example(s) how to use the model",
               "}")

    writeLines(lines, con = file.path(path, "man", paste0(name, ".Rd")))

    invisible(NULL)
}

create_model_run_man_file <- function(path, name)
{
    lines <- c("\\docType{methods}",
               paste0("\\name{run,", name, "-method}"),
               paste0("\\alias{run,", name, "-method}"),
               "\\title{Run the model}",
               "\\usage{",
               paste0("\\S4method{run}{", name, "}(model, threads = NULL, seed = NULL)"),
               "}",
               "\\arguments{",
               "\\item{model}{The model to run.}",
               "",
               "\\item{threads}{Number of threads. Default is NULL, i.e. to use all",
               "available processors.}",
               "",
               "\\item{seed}{Random number seed. Default is NULL, i.e. the",
               "simulator uses time to seed the random number generator.}",
               "}",
               "\\value{",
               "model with result from simulation.",
               "}",
               "\\description{",
               "Run the model",
               "}")

    writeLines(lines, con = file.path(path, "man", "run-methods.Rd"))

    invisible(NULL)
}

##' @rdname package_skeleton-methods
##' @export
setMethod("package_skeleton",
          signature(model = "SimInf_mparse"),
          function(model, name = NULL, path = ".", author = NULL,
                   email = NULL, maintainer = NULL, license = "GPL-3")
{
    stopifnot(!is.null(name), is.character(name), length(name) == 1,
              nchar(name) > 0)
    stopifnot(!is.null(path), is.character(path), length(path) == 1,
              nchar(path) > 0)
    path <- file.path(path, name)
    if (dir.exists(path))
        stop(paste0("'", path, "' already exists"))

    if (is.null(author)) {
        author <- "Your Name"
    } else {
        stopifnot(is.character(author), length(author) == 1, nchar(author) > 0)
    }

    if (is.null(email)) {
        email <- "your@email.com"
    } else {
        stopifnot(is.character(email), length(email) == 1, nchar(email) > 0)
    }

    if (is.null(maintainer)) {
        maintainer <- author
    } else {
        stopifnot(is.character(maintainer), length(maintainer) == 1,
                  nchar(maintainer) > 0)
    }

    ## Create folder structure
    message("Creating directories ...", domain = NA)
    dir.create(path, recursive = TRUE)
    dir.create(file.path(path, "man"))
    dir.create(file.path(path, "src"))
    dir.create(file.path(path, "R"))

    ## Create files
    message("Creating DESCRIPTION ...", domain = NA)
    create_DESCRIPTION_file(path, name, author, maintainer, email,
                            license)
    message("Creating NAMESPACE ...", domain = NA)
    create_NAMESPACE_file(path, name)
    message("Creating C file ...", domain = NA)
    create_model_C_file(path, model, name)
    message("Creating R file ...", domain = NA)
    create_model_R_file(path, model, name)
    message("Creating help files ...", domain = NA)
    create_model_man_file(path, name)
    create_model_class_man_file(path, name)
    create_model_run_man_file(path, name)

    invisible(NULL)
})
