#ifndef WHEEL_H
#define WHEEL_H

#include <Rcpp.h>

// This is the largest multiple of 2*3*5*7 = 210
// that is less than 2^15 = 32768 = 32KB. This
// is the typical size of most CPU's L1 cache
constexpr int_fast64_t Almost210L1Cache = 32760;

// This is the largest multiple of 2*3*5*7*11 = 2310
// that is less than 2^15 = 32768 = 32KB. This
// is the typical size of most CPU's L1 cache
constexpr int_fast64_t Almost2310L1Cache = 32340;

constexpr std::size_t NUM210 = 210;     //2 * 3 * 5 * 7
constexpr std::size_t NUM2310 = 2310;   //2 * 3 * 5 * 7 * 11
constexpr std::size_t NUM30030 = 30030; //2 * 3 * 5 * 7 * 11 * 13

// The number of relatively prime numbers to 210 less than 210
constexpr std::size_t SZ_WHEEL210 = 48;
// The number of relatively prime numbers to 2310 less than 2310
constexpr std::size_t SZ_WHEEL2310 = 480;
// The number of relatively prime numbers to 30030 less than 30030
constexpr std::size_t SZ_WHEEL30030 = 5760;

constexpr std::size_t N_WHEELS210_PER_SEG = static_cast<std::size_t>(Almost210L1Cache / NUM210);
constexpr std::size_t N_WHEELS2310_PER_SEG = static_cast<std::size_t>(Almost2310L1Cache / NUM2310);

static const int_fast64_t ARR_WHEEL210[SZ_WHEEL210] = {
    10, 2, 4, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4,
    2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4, 2, 10, 2};

static const int_fast64_t ARR_WHEEL2310[SZ_WHEEL2310] = {
    12, 4, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 4,
    14, 4, 6, 2, 10, 2, 6, 6, 4, 2, 4, 6, 2, 10, 2, 4, 2, 12, 10, 2, 4, 2, 4, 6,
    2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 6, 8, 6, 10, 2, 4, 6, 2,
    6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 6, 10, 2, 10, 2, 4, 2, 4, 6, 8, 4, 2, 4, 12,
    2, 6, 4, 2, 6, 4, 6, 12, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 10, 2, 4, 6,
    2, 6, 4, 2, 4, 2, 10, 2, 10, 2, 4, 6, 6, 2, 6, 6, 4, 6, 6, 2, 6, 4, 2, 6, 4,
    6, 8, 4, 2, 6, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 4, 2, 10, 2, 6, 4, 2, 4, 2, 10,
    2, 10, 2, 4, 2, 4, 8, 6, 4, 2, 4, 6, 6, 2, 6, 4, 8, 4, 6, 8, 4, 2, 4, 2, 4,
    8, 6, 4, 6, 6, 6, 2, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4, 2, 10, 2, 10, 2, 6, 4,
    6, 2, 6, 4, 2, 4, 6, 6, 8, 4, 2, 6, 10, 8, 4, 2, 4, 2, 4, 8, 10, 6, 2, 4, 8,
    6, 6, 4, 2, 4, 6, 2, 6, 4, 6, 2, 10, 2, 10, 2, 4, 2, 4, 6, 2, 6, 4, 2, 4, 6,
    6, 2, 6, 6, 6, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 8, 4, 6, 2, 6, 6, 4, 2, 4,
    6, 8, 4, 2, 4, 2, 10, 2, 10, 2, 4, 2, 4, 6, 2, 10, 2, 4, 6, 8, 6, 4, 2, 6,
    4, 6, 8, 4, 6, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 4, 6, 6, 2, 6, 6, 4, 2,
    10, 2, 10, 2, 4, 2, 4, 6, 2, 6, 4, 2, 10, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2,
    4, 2, 12, 6, 4, 6, 2, 4, 6, 2, 12, 4, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2, 10, 6,
    2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 10, 6, 8, 6, 4, 2, 4, 8, 6, 4, 6,
    2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 6, 4, 2, 4, 2, 10, 12, 2, 4, 2, 10, 2, 6, 4,
    2, 4, 6, 6, 2, 10, 2, 6, 4, 14, 4, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6,
    4, 2, 4, 6, 2, 6, 4, 2, 4, 12, 2};
    
static const int ARR_WHEEL30030[SZ_WHEEL30030] = {
    16, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 4, 14, 4,
    6, 2, 10, 2, 6, 6, 4, 6, 6, 2, 10, 2, 4, 2, 12, 12, 4, 2, 4, 6, 2, 10, 6, 6, 6,
    2, 6, 4, 2, 6, 4, 14, 4, 2, 4, 6, 8, 6, 10, 2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 6, 4,
    8, 10, 2, 10, 2, 4, 2, 4, 6, 8, 4, 2, 4, 12, 8, 4, 2, 6, 4, 6, 12, 2, 4, 2, 12,
    6, 4, 6, 6, 6, 2, 6, 10, 2, 4, 6, 2, 6, 6, 4, 2, 10, 2, 10, 2, 4, 6, 6, 2, 6, 6,
    4, 6, 8, 6, 4, 2, 6, 4, 6, 8, 4, 2, 6, 4, 8, 6, 4, 8, 4, 6, 8, 10, 2, 10, 2, 6,
    4, 2, 4, 2, 10, 2, 10, 2, 4, 2, 4, 14, 4, 2, 4, 6, 6, 2, 6, 4, 8, 10, 8, 4, 2, 4,
    6, 8, 6, 4, 6, 6, 6, 2, 6, 6, 4, 2, 4, 6, 2, 10, 2, 4, 2, 10, 2, 10, 2, 6, 4, 8,
    6, 4, 2, 4, 6, 6, 8, 4, 2, 6, 10, 8, 4, 2, 6, 4, 8, 10, 6, 2, 4, 8, 6, 6, 4, 2,
    4, 6, 2, 6, 4, 6, 2, 10, 12, 2, 4, 2, 4, 6, 2, 6, 4, 2, 4, 12, 2, 6, 6, 10, 6, 8,
    4, 2, 4, 2, 4, 8, 6, 12, 4, 6, 2, 12, 4, 2, 4, 6, 8, 4, 2, 4, 2, 12, 10, 2, 4, 2,
    4, 6, 2, 10, 2, 4, 6, 8, 6, 4, 2, 6, 4, 6, 8, 4, 6, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2,
    6, 6, 4, 6, 6, 8, 6, 4, 2, 10, 2, 10, 2, 4, 2, 10, 2, 6, 4, 2, 10, 6, 2, 6, 4, 2,
    6, 4, 6, 8, 6, 4, 2, 12, 10, 6, 2, 4, 6, 2, 12, 4, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2,
    10, 6, 2, 4, 6, 2, 6, 4, 2, 10, 6, 2, 6, 4, 12, 6, 8, 6, 4, 2, 4, 8, 6, 4, 6, 2,
    4, 6, 8, 6, 6, 4, 6, 2, 6, 4, 2, 4, 2, 10, 12, 2, 4, 12, 2, 6, 4, 2, 4, 6, 6, 2,
    12, 6, 4, 18, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 8, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4,
    12, 2, 12, 6, 4, 6, 2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 4, 14,
    4, 6, 2, 10, 2, 6, 6, 4, 2, 10, 2, 10, 2, 4, 14, 10, 2, 4, 2, 4, 6, 2, 6, 10, 6,
    6, 2, 10, 2, 6, 4, 6, 8, 4, 2, 4, 6, 8, 6, 10, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 2,
    6, 4, 2, 6, 10, 2, 10, 6, 2, 4, 6, 8, 4, 2, 4, 12, 2, 6, 4, 2, 6, 4, 6, 12, 2, 4,
    2, 4, 8, 6, 4, 6, 2, 10, 2, 6, 10, 6, 6, 2, 6, 4, 2, 4, 2, 10, 2, 12, 4, 6, 6, 2,
    12, 4, 6, 6, 2, 6, 4, 2, 6, 4, 14, 4, 2, 6, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 6, 10,
    2, 6, 4, 6, 2, 10, 2, 10, 2, 4, 2, 4, 8, 6, 4, 2, 4, 6, 6, 8, 4, 8, 4, 6, 8, 4,
    2, 4, 2, 12, 6, 4, 6, 6, 6, 2, 6, 6, 4, 2, 4, 6, 2, 6, 6, 4, 2, 10, 2, 10, 2, 6,
    4, 6, 2, 6, 4, 2, 4, 6, 14, 4, 2, 6, 10, 8, 4, 2, 4, 2, 4, 8, 10, 8, 4, 8, 6, 10,
    2, 4, 6, 2, 6, 4, 6, 2, 10, 2, 10, 2, 4, 2, 4, 6, 8, 4, 2, 4, 6, 6, 2, 6, 6, 6,
    10, 8, 4, 2, 4, 6, 8, 6, 4, 8, 4, 6, 2, 6, 6, 4, 2, 4, 6, 12, 2, 4, 2, 10, 2, 10,
    2, 4, 2, 4, 8, 10, 2, 4, 6, 8, 6, 4, 2, 6, 4, 6, 8, 4, 8, 4, 8, 6, 4, 6, 2, 4, 6,
    2, 6, 6, 4, 6, 6, 2, 6, 6, 4, 2, 10, 12, 2, 4, 2, 4, 6, 2, 6, 4, 2, 16, 2, 6, 4,
    2, 10, 6, 8, 4, 2, 4, 2, 12, 6, 10, 2, 4, 6, 2, 12, 4, 2, 4, 8, 6, 4, 2, 4, 2, 12,
    10, 6, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 10, 6, 8, 10, 2, 4, 8, 6, 4, 6,
    2, 4, 6, 2, 6, 6, 6, 4, 6, 8, 4, 2, 4, 2, 10, 12, 2, 4, 2, 10, 2, 6, 4, 2, 4, 6,
    6, 2, 10, 2, 6, 4, 14, 6, 4, 2, 4, 8, 10, 6, 2, 4, 6, 2, 6, 6, 4, 2, 4, 8, 6, 4,
    2, 4, 12, 2, 12, 4, 2, 4, 6, 2, 6, 4, 2, 10, 6, 2, 6, 4, 8, 4, 6, 8, 4, 2, 4, 2,
    4, 14, 4, 6, 2, 10, 8, 6, 4, 2, 4, 6, 2, 10, 2, 4, 2, 12, 10, 2, 4, 6, 6, 2, 6,
    4, 6, 6, 6, 2, 6, 6, 6, 4, 6, 12, 2, 4, 6, 8, 6, 10, 2, 4, 8, 6, 6, 4, 2, 4, 6,
    2, 6, 4, 2, 6, 10, 2, 10, 2, 6, 4, 6, 8, 4, 6, 12, 2, 6, 4, 2, 6, 4, 6, 12, 2, 4,
    2, 4, 14, 4, 6, 2, 4, 6, 2, 6, 10, 2, 10, 2, 6, 4, 2, 4, 12, 2, 10, 2, 4, 6, 6,
    2, 6, 6, 4, 6, 6, 2, 10, 2, 6, 4, 6, 8, 4, 2, 6, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 4,
    2, 10, 2, 6, 4, 2, 6, 10, 2, 10, 6, 2, 4, 8, 6, 4, 2, 4, 6, 6, 2, 6, 4, 8, 4, 6,
    8, 4, 2, 4, 2, 4, 8, 6, 4, 6, 12, 2, 6, 6, 4, 6, 6, 2, 6, 4, 2, 4, 2, 10, 2, 12,
    6, 4, 6, 2, 10, 2, 4, 6, 6, 8, 4, 2, 6, 18, 4, 2, 4, 2, 4, 8, 10, 6, 2, 4, 8, 6,
    6, 6, 4, 6, 2, 6, 4, 6, 2, 10, 2, 10, 2, 4, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 8, 6,
    6, 4, 6, 8, 4, 2, 4, 2, 12, 6, 4, 12, 6, 2, 6, 6, 4, 2, 4, 6, 8, 6, 4, 2, 10, 2,
    10, 2, 4, 2, 4, 6, 2, 10, 2, 4, 6, 8, 6, 4, 2, 6, 4, 6, 8, 4, 6, 2, 4, 8, 6, 4,
    8, 4, 6, 2, 6, 10, 6, 6, 2, 6, 6, 4, 2, 10, 2, 10, 2, 4, 2, 4, 6, 8, 4, 2, 10, 6,
    2, 6, 4, 2, 6, 10, 8, 4, 2, 4, 14, 6, 4, 6, 2, 4, 6, 2, 12, 4, 2, 4, 8, 10, 2, 4,
    2, 10, 2, 10, 6, 2, 4, 8, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 10, 6, 8, 6, 6, 4, 8, 6,
    4, 6, 2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 6, 4, 2, 4, 2, 10, 12, 2, 4, 2, 10, 2, 6, 4,
    2, 4, 12, 2, 10, 2, 10, 14, 4, 2, 4, 2, 4, 8, 6, 10, 2, 4, 6, 2, 12, 4, 2, 4, 6,
    2, 6, 4, 2, 4, 14, 12, 4, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8,
    4, 6, 2, 4, 14, 4, 6, 2, 10, 2, 6, 6, 4, 2, 4, 6, 12, 2, 4, 2, 12, 10, 2, 4, 2,
    10, 2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 6, 4, 6, 8, 16, 2, 4, 6, 2, 6, 6,
    4, 2, 4, 8, 6, 4, 2, 6, 10, 2, 10, 2, 4, 2, 4, 6, 8, 4, 2, 16, 2, 6, 4, 8, 4, 6,
    12, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 8, 10, 2, 4, 6, 2, 6, 4, 2, 4, 2, 10, 2, 10,
    2, 4, 6, 6, 2, 6, 6, 4, 6, 6, 2, 6, 6, 6, 4, 6, 12, 2, 6, 4, 8, 6, 4, 6, 2, 4, 14,
    6, 4, 2, 10, 2, 6, 4, 2, 4, 2, 10, 2, 10, 2, 6, 4, 8, 6, 4, 6, 6, 6, 2, 6, 4, 8,
    4, 6, 8, 4, 2, 4, 2, 4, 14, 4, 6, 6, 6, 2, 6, 6, 4, 2, 10, 2, 6, 4, 2, 4, 12, 2,
    10, 2, 6, 4, 6, 2, 6, 6, 4, 6, 6, 12, 2, 6, 10, 8, 4, 2, 4, 2, 4, 8, 10, 6, 2, 4,
    8, 6, 6, 4, 2, 4, 6, 2, 6, 4, 8, 10, 2, 10, 6, 2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 2,
    6, 6, 6, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 8, 10, 2, 6, 6, 4, 6, 6, 8, 4, 2, 4, 2,
    10, 2, 12, 4, 2, 4, 6, 2, 10, 2, 4, 6, 8, 6, 4, 2, 6, 4, 14, 4, 6, 2, 4, 8, 6, 4,
    6, 2, 4, 6, 2, 6, 6, 10, 6, 2, 6, 10, 2, 10, 2, 10, 2, 4, 2, 4, 6, 2, 6, 4, 2, 10,
    6, 8, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 12, 6, 4, 6, 6, 6, 2, 12, 4, 2, 4, 8, 6, 6,
    4, 2, 10, 2, 10, 6, 2, 4, 6, 2, 6, 4, 2, 4, 6, 8, 6, 4, 2, 10, 6, 8, 6, 4, 2, 4,
    8, 6, 4, 8, 4, 6, 2, 6, 12, 4, 6, 2, 6, 4, 2, 4, 2, 10, 12, 2, 4, 2, 10, 8, 4, 2,
    4, 6, 6, 2, 10, 2, 6, 18, 4, 2, 4, 6, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6,
    2, 10, 2, 4, 12, 2, 12, 4, 2, 4, 8, 6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4,
    2, 6, 4, 14, 4, 6, 2, 10, 2, 6, 6, 4, 2, 4, 6, 2, 10, 2, 4, 2, 22, 2, 4, 2, 4, 6,
    2, 6, 4, 6, 12, 2, 6, 4, 2, 10, 6, 8, 4, 2, 4, 6, 8, 6, 10, 2, 4, 6, 2, 12, 4, 2,
    4, 6, 2, 6, 4, 2, 6, 12, 10, 2, 4, 2, 4, 6, 8, 4, 2, 4, 12, 2, 6, 4, 2, 6, 4, 6,
    12, 6, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 10, 2, 4, 6, 8, 4, 2, 4, 2, 10, 2, 10, 2,
    4, 12, 2, 6, 6, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 6, 6, 4, 8, 10, 6, 2, 4, 6, 8,
    6, 4, 2, 12, 6, 4, 2, 4, 2, 10, 2, 10, 2, 4, 2, 4, 8, 6, 4, 2, 10, 6, 2, 6, 4, 8,
    4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 6, 6, 6, 8, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4, 2, 10,
    2, 10, 2, 10, 6, 2, 6, 4, 2, 4, 6, 6, 8, 6, 6, 10, 12, 2, 4, 2, 4, 8, 10, 6, 2,
    4, 8, 6, 6, 4, 2, 4, 6, 2, 6, 4, 6, 2, 10, 2, 10, 2, 6, 4, 6, 2, 6, 4, 6, 6, 6,
    2, 6, 6, 6, 4, 6, 8, 4, 2, 4, 2, 4, 14, 4, 8, 4, 6, 2, 6, 6, 4, 2, 10, 8, 4, 2,
    4, 12, 2, 10, 2, 4, 2, 4, 6, 2, 12, 4, 6, 8, 10, 2, 6, 4, 6, 8, 4, 6, 2, 4, 8, 6,
    4, 6, 2, 4, 6, 2, 6, 6, 4, 6, 6, 2, 6, 6, 6, 10, 2, 10, 6, 2, 4, 6, 2, 6, 4, 2,
    10, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 12, 6, 4, 6, 2, 10, 2, 12, 4, 6, 8, 6,
    4, 2, 4, 2, 10, 2, 16, 2, 4, 6, 2, 10, 2, 4, 6, 6, 2, 6, 4, 2, 10, 14, 6, 4, 2,
    4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 6, 4, 6, 2, 10, 12, 2, 4, 2, 10, 2,
    6, 4, 2, 4, 6, 6, 12, 2, 6, 4, 14, 4, 2, 4, 2, 12, 6, 4, 6, 6, 6, 2, 6, 6, 4, 2,
    4, 6, 2, 6, 6, 4, 12, 2, 12, 4, 2, 4, 6, 2, 6, 4, 2, 4, 6, 8, 6, 4, 2, 6, 4, 6,
    8, 4, 2, 4, 2, 4, 14, 4, 8, 10, 2, 6, 10, 2, 4, 6, 2, 10, 2, 4, 2, 12, 10, 2, 4,
    2, 4, 6, 8, 4, 6, 6, 6, 2, 6, 4, 2, 6, 10, 8, 4, 2, 4, 6, 8, 6, 10, 2, 4, 6, 2,
    6, 6, 4, 2, 4, 6, 2, 10, 2, 6, 10, 2, 10, 2, 4, 2, 4, 14, 4, 2, 4, 12, 2, 6, 4,
    2, 6, 4, 6, 12, 2, 6, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 10, 2, 4, 6, 2, 6, 4, 2, 4,
    2, 10, 12, 2, 4, 6, 6, 2, 6, 6, 4, 12, 2, 6, 4, 2, 10, 6, 8, 4, 2, 6, 4, 8, 6, 10,
    2, 4, 6, 14, 4, 2, 10, 2, 6, 4, 2, 4, 2, 12, 10, 2, 4, 2, 4, 8, 6, 4, 2, 4, 6, 6,
    2, 6, 4, 8, 4, 6, 8, 4, 6, 2, 4, 8, 6, 4, 6, 6, 6, 2, 6, 6, 4, 2, 4, 6, 8, 4, 2,
    4, 2, 10, 2, 10, 2, 6, 10, 2, 6, 4, 2, 4, 6, 6, 8, 4, 2, 6, 10, 8, 6, 4, 2, 4, 8,
    10, 6, 2, 4, 8, 6, 6, 4, 2, 4, 8, 6, 4, 6, 2, 10, 2, 10, 2, 4, 2, 4, 6, 2, 6, 4,
    2, 10, 6, 2, 6, 12, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 8, 4, 6, 8, 6, 4, 2, 4, 6,
    8, 4, 2, 4, 2, 10, 2, 10, 2, 4, 6, 6, 2, 10, 2, 4, 6, 8, 6, 6, 6, 4, 6, 12, 6, 2,
    4, 8, 6, 4, 6, 2, 4, 8, 6, 6, 4, 6, 6, 2, 6, 6, 4, 2, 10, 2, 10, 2, 6, 4, 6, 2,
    6, 4, 12, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 18, 4, 6, 2, 4, 6, 2, 12, 4, 2,
    12, 6, 4, 2, 4, 12, 2, 10, 6, 2, 4, 6, 2, 6, 6, 4, 6, 6, 2, 10, 2, 10, 6, 8, 6,
    4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 6, 4, 2, 6, 10, 12, 6, 2, 10,
    2, 6, 4, 2, 4, 6, 6, 2, 10, 2, 6, 4, 14, 4, 2, 4, 2, 4, 8, 6, 4, 6, 2, 10, 2, 6,
    6, 4, 6, 6, 2, 6, 4, 2, 4, 12, 2, 12, 4, 2, 4, 6, 2, 10, 2, 4, 6, 6, 2, 6, 4, 2,
    6, 4, 14, 4, 2, 4, 2, 4, 14, 4, 6, 2, 10, 2, 6, 6, 6, 4, 6, 2, 10, 6, 2, 12, 10,
    2, 4, 2, 4, 6, 2, 6, 4, 6, 6, 6, 8, 4, 2, 6, 4, 6, 8, 4, 2, 4, 14, 6, 10, 6, 6,
    2, 6, 6, 4, 2, 4, 6, 2, 6, 6, 6, 10, 2, 10, 2, 4, 2, 4, 6, 8, 4, 2, 4, 14, 6, 4,
    2, 6, 4, 6, 12, 2, 4, 2, 4, 8, 6, 4, 8, 4, 6, 2, 6, 10, 2, 4, 6, 2, 6, 4, 2, 4,
    2, 10, 2, 10, 2, 4, 6, 6, 8, 6, 4, 6, 6, 2, 6, 4, 2, 6, 10, 8, 4, 2, 10, 8, 6, 4,
    6, 2, 4, 6, 8, 6, 4, 2, 10, 2, 10, 2, 4, 2, 10, 2, 10, 2, 4, 2, 4, 8, 6, 4, 2, 4,
    6, 6, 2, 6, 4, 8, 4, 6, 8, 4, 2, 6, 4, 8, 6, 4, 6, 6, 6, 2, 6, 6, 4, 2, 4, 6, 2,
    6, 4, 2, 4, 2, 10, 12, 2, 6, 4, 6, 2, 6, 4, 2, 4, 12, 8, 4, 2, 16, 8, 4, 2, 4, 2,
    4, 8, 16, 2, 4, 8, 12, 4, 2, 4, 6, 2, 6, 4, 6, 2, 12, 10, 2, 4, 2, 4, 6, 2, 6, 4,
    2, 4, 6, 6, 2, 6, 6, 6, 4, 6, 8, 4, 6, 2, 4, 8, 6, 4, 8, 4, 6, 2, 6, 6, 4, 2, 4,
    6, 8, 4, 2, 4, 2, 10, 2, 10, 2, 4, 2, 10, 2, 10, 2, 4, 6, 8, 6, 4, 2, 6, 4, 6, 8,
    10, 2, 4, 8, 10, 6, 2, 4, 6, 2, 6, 6, 4, 6, 8, 6, 6, 4, 2, 10, 2, 10, 2, 4, 2, 4,
    6, 2, 6, 4, 2, 10, 6, 2, 6, 4, 8, 4, 6, 8, 4, 2, 4, 2, 12, 6, 4, 6, 2, 4, 6, 14,
    4, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2, 10, 6, 6, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 6, 10,
    6, 14, 4, 2, 4, 8, 6, 4, 6, 2, 4, 8, 6, 6, 6, 4, 6, 2, 6, 4, 2, 4, 2, 10, 12, 2,
    6, 10, 2, 6, 4, 6, 6, 6, 2, 10, 2, 6, 4, 14, 4, 2, 4, 2, 4, 14, 4, 6, 2, 4, 6, 2,
    6, 6, 4, 2, 10, 2, 6, 4, 2, 4, 12, 2, 12, 4, 2, 4, 6, 2, 6, 6, 4, 6, 6, 2, 10, 2,
    6, 4, 6, 8, 4, 2, 4, 2, 4, 14, 4, 6, 2, 10, 2, 6, 6, 4, 2, 4, 6, 2, 10, 2, 6, 12,
    10, 6, 2, 4, 6, 2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 6, 8, 6, 10,
    2, 10, 2, 6, 6, 4, 6, 6, 2, 6, 4, 2, 6, 10, 2, 12, 4, 2, 4, 6, 12, 2, 4, 12, 2,
    6, 4, 2, 6, 4, 18, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 12, 4, 6, 2, 6, 4, 6,
    2, 10, 2, 10, 2, 4, 6, 6, 2, 6, 6, 4, 6, 6, 8, 4, 2, 6, 4, 6, 8, 4, 2, 6, 12, 6,
    4, 6, 6, 6, 8, 6, 4, 2, 10, 2, 6, 6, 4, 2, 10, 2, 10, 2, 4, 2, 4, 8, 6, 4, 2, 4,
    6, 8, 6, 4, 8, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 12, 6, 2, 6, 10, 2, 4, 6, 2, 6,
    4, 2, 4, 2, 10, 2, 10, 2, 6, 4, 6, 8, 4, 2, 4, 6, 6, 8, 4, 2, 6, 10, 8, 4, 2, 4,
    6, 8, 10, 6, 2, 4, 8, 6, 6, 4, 2, 4, 6, 2, 10, 6, 2, 10, 2, 10, 2, 4, 2, 4, 8, 6,
    4, 2, 4, 6, 6, 2, 6, 6, 6, 4, 6, 8, 4, 2, 6, 4, 8, 6, 4, 8, 4, 6, 2, 6, 6, 4, 2,
    4, 6, 8, 4, 2, 4, 2, 10, 12, 2, 4, 2, 4, 6, 2, 10, 2, 4, 14, 6, 4, 2, 10, 6, 8,
    4, 6, 2, 4, 8, 6, 10, 2, 4, 6, 2, 12, 4, 6, 6, 2, 6, 6, 4, 2, 12, 10, 2, 4, 2, 4,
    6, 2, 6, 4, 2, 10, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 6, 2, 12, 6, 4, 6, 2, 4, 6, 2,
    12, 4, 2, 4, 14, 4, 2, 4, 2, 10, 2, 10, 6, 2, 10, 2, 6, 4, 2, 4, 6, 6, 2, 6, 4,
    2, 10, 6, 8, 6, 4, 2, 4, 8, 10, 6, 2, 4, 6, 2, 6, 6, 6, 4, 8, 6, 4, 2, 4, 2, 10,
    12, 2, 4, 2, 10, 2, 6, 4, 2, 10, 6, 2, 10, 8, 4, 14, 4, 2, 4, 2, 4, 8, 6, 4, 6,
    2, 4, 6, 8, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4, 12, 2, 12, 4, 6, 6, 2, 6, 4, 2, 4, 6,
    6, 2, 6, 6, 6, 4, 6, 12, 2, 4, 2, 4, 14, 4, 6, 2, 12, 6, 6, 4, 2, 4, 6, 2, 10, 2,
    4, 2, 12, 10, 2, 6, 4, 6, 2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 6,
    14, 10, 2, 4, 6, 2, 6, 6, 4, 2, 10, 2, 6, 4, 2, 16, 2, 10, 2, 4, 2, 4, 6, 8, 6,
    4, 12, 2, 10, 2, 6, 4, 6, 12, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 10, 2, 4, 6,
    2, 6, 4, 2, 6, 10, 2, 10, 6, 6, 6, 2, 6, 6, 4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4,
    2, 6, 4, 8, 6, 4, 6, 2, 10, 8, 6, 4, 12, 2, 6, 4, 2, 4, 2, 10, 2, 12, 4, 2, 4, 8,
    10, 2, 4, 6, 6, 2, 6, 4, 8, 4, 14, 4, 2, 4, 2, 4, 8, 6, 4, 6, 6, 6, 2, 6, 6, 6,
    4, 6, 2, 6, 4, 6, 2, 10, 2, 10, 2, 6, 4, 6, 2, 6, 4, 2, 4, 6, 6, 8, 4, 2, 6, 10,
    8, 4, 2, 4, 2, 12, 10, 6, 6, 8, 6, 6, 4, 2, 4, 6, 2, 6, 10, 2, 10, 2, 10, 2, 4,
    2, 4, 6, 2, 6, 4, 2, 4, 6, 8, 6, 6, 6, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 8, 4, 6,
    2, 6, 10, 2, 4, 6, 8, 4, 2, 4, 2, 10, 2, 10, 2, 4, 2, 4, 6, 12, 2, 4, 6, 8, 6, 4,
    2, 6, 10, 8, 4, 6, 6, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 4, 6, 6, 2, 12, 4, 2, 10, 2,
    10, 2, 4, 2, 4, 8, 6, 4, 2, 10, 6, 2, 6, 4, 2, 6, 4, 6, 8, 4, 2, 6, 12, 6, 4, 6,
    2, 4, 6, 2, 12, 4, 2, 4, 8, 6, 4, 2, 4, 2, 10, 12, 6, 2, 4, 6, 2, 6, 4, 2, 4, 12,
    2, 6, 4, 2, 10, 6, 8, 6, 4, 2, 4, 8, 6, 10, 2, 4, 6, 2, 12, 6, 4, 6, 2, 6, 4, 2,
    4, 2, 22, 2, 4, 2, 10, 2, 6, 4, 2, 4, 6, 6, 2, 10, 2, 6, 4, 14, 4, 6, 2, 4, 8, 6,
    4, 6, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 8, 4, 2, 4, 12, 2, 12, 4, 2, 10, 2, 6, 4, 2,
    4, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8, 6, 4, 2, 4, 18, 6, 2, 10, 2, 6, 6, 4, 2, 4, 8,
    10, 2, 4, 2, 12, 10, 2, 4, 2, 4, 6, 2, 6, 4, 12, 6, 2, 6, 4, 8, 4, 6, 8, 4, 2, 4,
    6, 8, 6, 10, 2, 4, 6, 8, 6, 4, 2, 4, 6, 2, 6, 4, 2, 6, 10, 2, 10, 2, 4, 6, 6, 8,
    4, 2, 4, 12, 2, 6, 6, 6, 4, 6, 12, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 8, 6, 10, 2, 4,
    6, 2, 6, 4, 2, 4, 2, 10, 2, 10, 2, 10, 6, 2, 6, 10, 6, 6, 2, 6, 4, 2, 6, 4, 6, 8,
    4, 2, 6, 4, 14, 4, 6, 2, 4, 6, 8, 6, 4, 2, 10, 2, 6, 4, 2, 4, 12, 2, 10, 2, 4, 2,
    4, 8, 6, 6, 4, 6, 6, 2, 10, 8, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6, 4, 6, 6, 6, 2, 6, 6,
    4, 2, 4, 6, 2, 6, 4, 2, 6, 10, 2, 10, 8, 4, 6, 2, 6, 4, 2, 4, 6, 6, 8, 4, 2, 6,
    10, 8, 4, 2, 4, 2, 4, 8, 10, 6, 2, 12, 6, 6, 4, 6, 6, 2, 6, 4, 6, 2, 10, 2, 12,
    4, 2, 4, 6, 2, 10, 2, 4, 6, 6, 2, 6, 6, 6, 4, 14, 4, 2, 4, 2, 4, 8, 6, 4, 8, 4,
    6, 2, 6, 6, 6, 4, 6, 8, 4, 6, 2, 10, 2, 10, 2, 4, 2, 4, 6, 2, 10, 2, 4, 6, 14, 4,
    2, 6, 4, 6, 8, 4, 6, 2, 12, 6, 4, 6, 6, 6, 2, 6, 6, 4, 6, 6, 2, 6, 6, 4, 2, 10,
    2, 10, 2, 4, 2, 4, 6, 2, 6, 4, 2, 10, 8, 6, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 12, 6,
    4, 8, 4, 6, 2, 16, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2, 10, 6, 2, 4, 6, 8, 4, 2, 4, 6,
    6, 2, 6, 4, 2, 16, 8, 6, 4, 6, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 10, 2,
    4, 2, 10, 12, 2, 4, 2, 12, 6, 4, 2, 4, 6, 6, 2, 10, 2, 6, 4, 14, 4, 2, 6, 4, 8,
    6, 4, 6, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4, 12, 14, 4, 2, 4, 6, 2, 6,
    4, 2, 4, 12, 2, 6, 4, 2, 10, 6, 8, 4, 2, 4, 2, 4, 14, 10, 2, 10, 2, 12, 4, 2, 4,
    6, 2, 10, 2, 4, 2, 12, 10, 2, 4, 2, 4, 6, 2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6, 4, 6,
    8, 4, 6, 6, 8, 6, 10, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 8, 4, 2, 6, 10, 2, 10, 2, 4,
    2, 10, 8, 4, 2, 4, 12, 2, 6, 4, 2, 6, 4, 6, 14, 4, 2, 4, 8, 10, 6, 2, 4, 6, 2, 6,
    10, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2, 10, 2, 4, 6, 6, 2, 6, 6, 10, 6, 2, 6, 4, 8, 4,
    6, 8, 4, 2, 6, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 4, 2, 10, 2, 6, 4, 2, 4, 2, 10, 2,
    10, 2, 4, 6, 8, 6, 4, 2, 4, 6, 6, 2, 6, 12, 4, 6, 12, 2, 4, 2, 4, 8, 6, 4, 6, 6,
    8, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 4, 2, 10, 2, 10, 2, 6, 4, 6, 2, 6, 4, 6, 6, 6,
    8, 4, 2, 6, 10, 8, 4, 2, 4, 2, 4, 18, 6, 2, 4, 8, 6, 6, 4, 2, 10, 2, 6, 4, 6, 12,
    2, 10, 2, 4, 2, 4, 6, 2, 6, 6, 4, 6, 6, 2, 12, 6, 4, 6, 8, 4, 2, 4, 2, 4, 8, 6,
    4, 8, 4, 6, 2, 6, 6, 4, 2, 4, 6, 8, 4, 2, 6, 10, 2, 10, 6, 2, 4, 6, 2, 10, 2, 4,
    6, 8, 6, 4, 2, 6, 4, 6, 8, 4, 6, 2, 4, 8, 6, 4, 6, 2, 10, 2, 6, 6, 4, 6, 6, 2, 6,
    6, 4, 2, 10, 2, 12, 4, 2, 4, 6, 2, 10, 2, 10, 6, 2, 6, 4, 2, 6, 4, 14, 4, 2, 4,
    2, 12, 6, 4, 6, 2, 4, 6, 2, 12, 6, 4, 8, 6, 4, 6, 2, 10, 2, 10, 6, 2, 4, 6, 2, 6,
    4, 2, 4, 6, 6, 8, 4, 2, 10, 6, 8, 6, 4, 2, 12, 6, 4, 6, 6, 6, 2, 6, 6, 6, 4, 6,
    2, 6, 6, 4, 2, 10, 12, 2, 4, 2, 10, 2, 6, 4, 2, 4, 6, 8, 10, 2, 6, 4, 14, 4, 2,
    4, 2, 4, 8, 6, 4, 8, 4, 6, 2, 6, 10, 2, 4, 6, 2, 6, 4, 2, 4, 12, 2, 12, 4, 2, 4,
    6, 8, 4, 2, 4, 6, 6, 2, 6, 4, 2, 6, 10, 8, 4, 2, 4, 6, 14, 4, 6, 2, 10, 2, 6, 6,
    4, 2, 4, 6, 2, 10, 2, 4, 2, 12, 10, 2, 4, 2, 4, 8, 6, 4, 6, 6, 6, 2, 6, 4, 2, 6,
    4, 6, 8, 4, 2, 10, 8, 6, 10, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 6, 10, 12,
    2, 4, 2, 4, 6, 8, 4, 2, 4, 12, 2, 6, 4, 2, 10, 6, 12, 2, 4, 2, 4, 8, 6, 10, 2, 4,
    6, 2, 16, 2, 4, 6, 2, 6, 4, 2, 4, 2, 12, 10, 2, 4, 6, 6, 2, 6, 6, 4, 6, 6, 2, 6,
    4, 2, 6, 4, 6, 8, 4, 8, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 4, 2, 10, 8, 4, 2, 4, 2, 10,
    2, 10, 2, 4, 2, 12, 6, 4, 2, 4, 6, 6, 2, 6, 4, 8, 4, 6, 8, 6, 4, 2, 4, 8, 10, 6,
    6, 6, 2, 6, 6, 4, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2, 10, 2, 6, 4, 6, 2, 6, 4, 2, 10,
    6, 8, 4, 8, 10, 8, 4, 2, 4, 2, 4, 8, 10, 6, 2, 4, 14, 6, 4, 2, 4, 6, 2, 6, 4, 6,
    2, 10, 2, 10, 2, 4, 6, 6, 2, 6, 4, 2, 4, 6, 6, 2, 6, 6, 6, 4, 6, 12, 2, 4, 2, 4,
    8, 6, 4, 8, 4, 8, 6, 6, 4, 2, 4, 6, 8, 4, 2, 4, 2, 10, 2, 10, 2, 6, 4, 6, 2, 10,
    6, 6, 8, 6, 4, 2, 6, 4, 6, 8, 4, 6, 2, 4, 14, 4, 6, 2, 4, 6, 2, 6, 6, 4, 12, 2,
    6, 6, 4, 12, 2, 10, 2, 4, 2, 4, 6, 2, 6, 6, 10, 6, 2, 10, 2, 6, 4, 6, 8, 4, 2, 4,
    2, 12, 6, 4, 6, 2, 4, 6, 2, 12, 4, 2, 4, 8, 6, 4, 2, 6, 10, 2, 10, 6, 2, 4, 6, 2,
    6, 4, 2, 4, 6, 6, 2, 6, 4, 2, 10, 6, 8, 6, 4, 2, 4, 8, 6, 4, 6, 2, 10, 2, 6, 6,
    10, 6, 2, 6, 4, 2, 4, 2, 10, 14, 4, 2, 10, 2, 10, 2, 4, 6, 6, 2, 10, 2, 6, 4, 14,
    4, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 6, 4, 6, 2, 6, 4, 6, 12, 2, 12, 4,
    2, 4, 6, 2, 6, 4, 2, 4, 6, 6, 8, 4, 2, 6, 4, 6, 8, 4, 2, 4, 2, 18, 4, 6, 12, 2,
    6, 6, 4, 2, 4, 6, 2, 12, 4, 2, 12, 10, 2, 4, 2, 4, 6, 2, 6, 4, 6, 6, 8, 6, 4, 2,
    6, 4, 6, 8, 4, 2, 4, 6, 8, 6, 12, 4, 6, 2, 6, 10, 2, 4, 6, 2, 6, 4, 2, 6, 10, 2,
    10, 2, 4, 2, 4, 6, 8, 4, 2, 4, 12, 2, 6, 4, 2, 6, 10, 12, 2, 4, 6, 8, 6, 4, 6, 2,
    4, 6, 2, 6, 10, 2, 4, 6, 2, 10, 2, 4, 2, 10, 2, 10, 2, 4, 6, 8, 6, 6, 4, 6, 6, 2,
    6, 4, 2, 6, 4, 6, 8, 4, 2, 6, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 4, 2, 10, 2, 6, 4, 2,
    4, 2, 10, 12, 2, 4, 2, 4, 8, 6, 4, 2, 4, 12, 2, 6, 4, 12, 6, 8, 4, 2, 4, 2, 4, 8,
    6, 10, 6, 6, 2, 12, 4, 2, 4, 6, 2, 6, 4, 2, 4, 2, 12, 10, 2, 6, 4, 6, 2, 6, 4, 2,
    4, 6, 6, 8, 4, 2, 6, 10, 8, 4, 6, 2, 4, 8, 10, 6, 2, 4, 8, 6, 6, 4, 2, 4, 6, 8,
    4, 6, 2, 10, 2, 10, 2, 4, 2, 10, 2, 6, 4, 2, 4, 6, 6, 2, 6, 6, 6, 4, 6, 8, 6, 4,
    2, 4, 8, 10, 8, 4, 6, 2, 6, 6, 4, 2, 4, 14, 4, 2, 4, 2, 10, 2, 10, 2, 4, 2, 4, 6,
    2, 10, 2, 10, 8, 6, 4, 8, 4, 6, 8, 4, 6, 2, 4, 8, 6, 4, 6, 2, 4, 6, 8, 6, 4, 6,
    6, 2, 6, 6, 4, 2, 10, 2, 10, 2, 4, 6, 6, 2, 6, 4, 2, 10, 6, 2, 6, 6, 6, 4, 6, 12,
    2, 4, 2, 12, 6, 4, 6, 2, 4, 8, 12, 4, 2, 4, 8, 6, 4, 2, 4, 2, 10, 2, 10, 8, 4, 6,
    2, 6, 4, 6, 6, 6, 2, 6, 4, 2, 10, 6, 8, 6, 4, 2, 4, 14, 4, 6, 2, 4, 6, 2, 6, 6,
    6, 10, 2, 6, 4, 2, 4, 12, 12, 2, 4, 2, 10, 2, 6, 6, 4, 6, 6, 2, 10, 2, 6, 4, 14,
    4, 2, 4, 2, 4, 8, 6, 4, 6, 2, 4, 6, 2, 6, 6, 4, 2, 4, 6, 2, 6, 4, 2, 16, 2};

#endif
