% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/origin_helper.r, R/origin_methods.r
\name{origin}
\alias{origin}
\alias{origin_backtracking}
\alias{origin_centrality}
\alias{origin_edm}
\title{Origin Estimation for Propagation Processes on Complex Networks}
\usage{
origin(events, type = c("edm", "backtracking", "centrality"), ...)

origin_edm(events, distance, silent = TRUE)

origin_backtracking(events, graph, start_with_event_node = TRUE,
  silent = TRUE)

origin_centrality(events, graph, silent = TRUE)
}
\arguments{
\item{events}{numeric vector of event counts at a specific time point}

\item{type}{character specifying the method, \code{'edm'}, \code{'backtracking'} and \code{'centrality'} are available.}

\item{...}{parameters to be passed to origin methods \code{\link{origin_edm}}, \code{\link{origin_backtracking}} or \code{\link{origin_centrality}}}

\item{distance}{numeric matrix specifying the distance matrix (for \code{type='edm'})}

\item{silent}{locigal, should the messages be suppressed?}

\item{graph}{igraph object specifying the underlying network graph (for \code{type='backtracking'} and \code{type='centrality'})}

\item{start_with_event_node}{logical specifying whether backtracking only starts from nodes that experienced events (for \code{type='backtracking'})}
}
\value{
\code{origin_edm} returns an object of class \code{origin}, list with 
\itemize{
 \item \code{est} origin estimate
 \item \code{aux} \code{data.frame} with auxiliary variables 
   \itemize{
      \item \code{id} as node identifier, 
      \item \code{events} for event magnitude, 
      \item \code{wmean} for weighted mean, 
      \item \code{wvar} for weighted variance, and 
      \item \code{mdist} mean distance from a node to all other nodes.
   }
 \item \code{type = 'edm'} effective distance median origin estimation
 }

\code{origin_backtracking} returns an object of class \code{origin}, list with 
\itemize{
 \item \code{est} origin estimate
 \item \code{aux} \code{data.frame} with auxiliary variables 
   \itemize{
      \item \code{id} as node identifier, 
      \item \code{events} for event magnitude, and
      \item \code{bcount} for backtracking counts, how often backtracking identifies this source node.
   }
 \item \code{type = 'backtracking'} backtracking origin estimation
 }

\code{origin_centrality} returns an object of class \code{origin}, list with 
\itemize{
 \item \code{est} origin estimate
 \item \code{aux} \code{data.frame} with auxiliary variables 
   \itemize{
      \item \code{id} as node identifier, 
      \item \code{events} for event magnitude, and
      \item \code{cent} for node centrality (betweenness divided degree).
   }
 \item \code{type = 'centrality'} centrality-based origin estimation
 }
}
\description{
This is the main function for origin estimation for propagation processes on complex networks. Different methods are available: effective distance median (\code{'edm'}), recursive backtracking (\code{'backtracking'}), and centrality-based source estimation (\code{'centrality'}).
For details on the methodological background, we refer to the corresponding publications.

\code{origin_edm} for effective distance-median origin estimation (Manitz et al., 2016)

\code{origin_backtracking} for recursive backtracking origin estimation (Manitz et al., 2016)

\code{origin_centrality} for centrality-based origin estimation (Comin et al., 2011)
}
\examples{
data(delayGoe)

# compute effective distance
data(ptnGoe)
goenet <- igraph::as_adjacency_matrix(ptnGoe, sparse=FALSE)
p <- goenet/rowSums(goenet)
eff <- eff_dist(p)
# apply effective distance median source estimation
om <- origin(events=delayGoe[10,-c(1:2)], type='edm', distance=eff)
summary(om)
plot(om, 'mdist',start=1)
plot(om, 'wvar',start=1)
performance(om, start=1, graph=ptnGoe)

# backtracking origin estimation (Manitz et al., 2016)
ob <- origin(events=delayGoe[10,-c(1:2)], type='backtracking', graph=ptnGoe)
summary(ob)
plot(ob, start=1)
performance(ob, start=1, graph=ptnGoe)

# centrality-based origin estimation (Comin et al., 2011)
oc <- origin(events=delayGoe[10,-c(1:2)], type='centrality', graph=ptnGoe)
summary(oc)
plot(oc, start=1)
performance(oc, start=1, graph=ptnGoe)

}
\author{
Juliane Manitz with contributions by Jonas Harbering
}
\references{
\itemize{
  \item Comin, C. H. and da Fontoura Costa, L. (2011). Identifying the starting point of a spreading process in complex networks. Physical Review E, 84. <DOI: 10.1103/PhysRevE.84.056105>
  \item Manitz, J., J. Harbering, M. Schmidt, T. Kneib, and A. Schoebel (2016). Source Estimation for Propagation Processes on Complex Networks with an Application to Delays in Public Transportation Systems. Accepted at JRSS-C.
 \item Manitz, J. (2014). Statistical Inference for Propagation Processes on Complex Networks. Ph.D. thesis, Georg-August-University Goettingen. Verlag Dr.~Hut, ISBN 978-3-8439-1668-4. Available online: \url{http://ediss.uni-goettingen.de/handle/11858/00-1735-0000-0022-5F38-B}.
  \item Manitz, J., Kneib, T., Schlather, M., Helbing, D. and Brockmann, D. (2014). Origin detection during food-borne disease outbreaks - a case study of the 2011 EHEC/HUS outbreak in Germany. PLoS Currents Outbreaks, 1. <DOI: 10.1371/currents.outbreaks.f3fdeb08c5b9de7c09ed9cbcef5f01f2>
}
}
\seealso{
Other origin-est: \code{\link{origin_multiple}}
}

