% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connectivity_estimation.finite_settlement.R
\encoding{UTF-8}
\name{d.rel.conn.finite.settlement}
\alias{d.rel.conn.finite.settlement}
\alias{p.rel.conn.finite.settlement}
\alias{q.rel.conn.finite.settlement}
\title{Estimate the probability distribution for the number of settlers originating 
at a site given a sample from a finite settler pool}
\usage{
d.rel.conn.finite.settlement(n.origin, p, k, n.obs, n.settlers,
  prior.n.origin = 1)

p.rel.conn.finite.settlement(n.origin, p, k, n.obs, n.settlers,
  prior.n.origin = 1)

q.rel.conn.finite.settlement(q, p, k, n.obs, n.settlers, prior.n.origin = 1)
}
\arguments{
\item{n.origin}{Vector of integers of possible numbers of settlers in the 
cohort that originated at the site of marking. All values should be 
integers \code{<=n.settlers}.}

\item{p}{Fraction of individuals (i.e., eggs) marked in the source population}

\item{k}{Number of marked settlers in sample}

\item{n.obs}{Total number of settlers collected}

\item{n.settlers}{Total number of settlers at the destination site from which
the \code{n.obs} (\code{<=n.settlers}) settlers are collected}

\item{prior.n.origin}{A prior probability mass function for the number of 
settlers in the cohort originating at the site of marking. Must be a scalar
or a vector of length \code{n.settlers+1}. Defaults to \code{1}.}

\item{q}{Vector of quantiles}
}
\value{
A vector of probabilities or quantiles.
}
\description{
These functions calculate the probability mass function 
(\code{d.rel.conn.finite.settlement}), the cumulative distribution function 
(\code{p.rel.conn.finite.settlement}) and the quantile function 
(\code{q.rel.conn.finite.settlement}) for the true number of settlers at a 
site that originated in a particular site given a known fraction of marked 
eggs among the eggs originating at the source site, a sample of settlers at 
the destination site, a known fraction of which are marked, and a finite 
settler pool of known size.
}
\details{
The relative connectivity between the source and destination sites is 
calculated as \code{n.origin/n.settlers}.
}
\section{Functions}{
\itemize{
\item \code{d.rel.conn.finite.settlement}: Returns the probability mass 
function for the numbers of settlers in the cohort that originated at the
source site (i.e., site of marking).

\item \code{p.rel.conn.finite.settlement}: Returns the cumulative distribution 
function for the numbers of settlers in the cohort that originated at the
source site (i.e., site of marking).

\item \code{q.rel.conn.finite.settlement}: Returns quantiles of the cumulative
distribution function for the numbers of settlers in the cohort that
originated at the source site (i.e., site of marking).
}}
\examples{
library(ConnMatTools)

k <- 10 # Number of marked settlers among sample
n.obs <- 87 # Number of settlers in sample
n.settlers <- 100 # Total size of settler pool

p <- 0.4 # Fraction of eggs that was marked
phi <- seq(0,1,length.out=101) # Values for relative connectivity

# Probability distribution assuming infinite settler pool and uniform prior
drc <- d.rel.conn.unif.prior(phi,p,k,n.obs)
prc <- p.rel.conn.unif.prior(phi,p,k,n.obs)
qrc <- q.rel.conn.unif.prior(c(0.025,0.975),p,k,n.obs) # 95\% confidence interval

# Test with finite settlement function and large (approx. infinite) settler pool
# Can be a bit slow for large settler pools
dis <- d.rel.conn.finite.settlement(0:(7*n.obs),p,k,n.obs,7*n.obs)

# Quantiles
qis <- q.rel.conn.finite.settlement(c(0.025,0.975),p,k,n.obs,7*n.obs)

# Finite settler pool
dfs <- d.rel.conn.finite.settlement(0:n.settlers,p,k,n.obs,n.settlers)

# Quantiles for the finite settler pool
qfs <- q.rel.conn.finite.settlement(c(0.025,0.975),p,k,n.obs,n.settlers)

# Make a plot of different distributions
plot(phi,drc,type="l",main="Probability of relative connectivity values",
     xlab=expression(phi),ylab="Probability density")
lines(phi,prc,col="blue")
lines((0:(7*n.obs))/(7*n.obs),dis*(7*n.obs),col="black",lty="dashed")
lines((0:n.settlers)/n.settlers,dfs*n.settlers,col="red",lty="dashed")
abline(v=qrc,col="black")
abline(v=qis/(7*n.obs),col="black",lty="dashed")
abline(v=qfs/n.settlers,col="red",lty="dashed")

}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\references{
Kaplan DM, Cuif M, Fauvelot C, Vigliola L, Nguyen-Huu T, Tiavouane J and Lett C 
  (in press) Uncertainty in empirical estimates of marine larval connectivity. 
  ICES Journal of Marine Science. doi:10.1093/icesjms/fsw182.
}
\seealso{
Other connectivity.estimation: \code{\link{d.rel.conn.beta.prior}},
  \code{\link{d.rel.conn.dists.func}},
  \code{\link{d.rel.conn.multinomial.unnorm}},
  \code{\link{d.rel.conn.multiple}},
  \code{\link{d.rel.conn.unif.prior}},
  \code{\link{dual.mark.transmission}},
  \code{\link{optim.rel.conn.dists}},
  \code{\link{r.marked.egg.fraction}}
}

