% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ms_graph.R
\docType{class}
\name{ms_graph}
\alias{ms_graph}
\title{Microsoft Graph}
\format{
An R6 object of class \code{ms_graph}.
}
\description{
Base class for interacting with Microsoft Graph API.
}
\section{Methods}{

\itemize{
\item \code{new(tenant, app, ...)}: Initialize a new Microsoft Graph connection with the given credentials. See 'Authentication' for more details.
\item \code{create_app(name, ..., add_password=TRUE, password_name=NULL, password_duration=2, certificate=NULL, create_service_principal=TRUE)}: Creates a new app registration in Azure Active Directory. See 'App creation' below.
\item \code{get_app(app_id, object_id)}: Retrieves an existing app registration, via either its app ID or object ID.
\item \code{list_apps(filter=NULL, n=Inf)}: Lists the app registrations in the current tenant.
\item \code{delete_app(app_id, object_id, confirm=TRUE)}: Deletes an existing app registration. Any associated service principal will also be deleted.
\item \code{create_service_principal(app_id, ...)}: Creates a service principal for a app registration.
\item \code{get_service_principal()}: Retrieves an existing service principal.
\item \code{list_service_principals(filter=NULL, n=Inf)}: Lists the service principals in the current tenant.
\item \code{delete_service_principal()}: Deletes an existing service principal.
\item \code{create_user(name, email, enabled=TRUE, ..., password=NULL, force_password_change=TRUE)}: Creates a new user account. By default this will be a work account (not social or local) in the current tenant, and will have a randomly generated password that must be changed at next login.
\item \code{get_user(user_id, email, name)}: Retrieves an existing user account. You can supply either the user ID, email address, or display name. The default is to return the logged-in user.
\item \code{list_users(filter=NULL, n=Inf)}: Lists the users in the current tenant.
\item \code{delete_user(user_id, email, name, confirm=TRUE)}: Deletes a user account.
\item \code{create_group(name, email, ...)}: Creates a new group. Note that only security groups can be created via the Microsoft Graph API.
\item \code{get_group(group_id, name)}: Retrieves an existing group.
\item \code{list_groups(filter=NULL, n=Inf)}: Lists the groups in the current tenant.
\item \code{delete_group(group_id, name, confirm=TRUE)}: Deletes a group.
\item \code{call_graph_endpoint(op="", ...)}: Calls the Microsoft Graph API using this object's token and tenant as authentication arguments. See \link{call_graph_endpoint}.
\item \code{call_batch_endpoint(requests=list(), ...)}: Calls the batch endpoint with a list of individual requests. See \link{call_batch_endpoint}.
\item \code{get_aad_object(id)}: Retrieves an arbitrary Azure Active Directory object by ID.
}
}

\section{Authentication}{

The recommended way to authenticate with Microsoft Graph is via the \link{create_graph_login} function, which creates a new instance of this class.

To authenticate with the \code{ms_graph} class directly, provide the following arguments to the \code{new} method:
\itemize{
\item \code{tenant}: Your tenant ID. This can be a name ("myaadtenant"), a fully qualified domain name ("myaadtenant.onmicrosoft.com" or "mycompanyname.com"), or a GUID.
\item \code{app}: The client/app ID to use to authenticate with Azure Active Directory. The default is to login interactively using the Azure CLI cross-platform app, but it's recommended to supply your own app credentials if possible.
\item \code{password}: if \code{auth_type == "client_credentials"}, the app secret; if \code{auth_type == "resource_owner"}, your account password.
\item \code{username}: if \code{auth_type == "resource_owner"}, your username.
\item \code{certificate}: If `auth_type == "client_credentials", a certificate to authenticate with. This is a more secure alternative to using an app secret.
\item \code{auth_type}: The OAuth authentication method to use, one of "client_credentials", "authorization_code", "device_code" or "resource_owner". See \link{get_azure_token} for how the default method is chosen, along with some caveats.
\item \code{version}: The Azure Active Directory (AAD) version to use for authenticating.
\item \code{host}: your Microsoft Graph host. Defaults to \verb{https://graph.microsoft.com/}.
\item \code{aad_host}: Azure Active Directory host for authentication. Defaults to \verb{https://login.microsoftonline.com/}. Change this if you are using a government or private cloud.
\item \code{scopes}: The Microsoft Graph scopes (permissions) to obtain for this Graph login. Only for \code{version=2}.
\item \code{token}: Optionally, an OAuth 2.0 token, of class \link[AzureAuth:AzureToken]{AzureAuth::AzureToken}. This allows you to reuse the authentication details for an existing session. If supplied, all other arguments will be ignored.
}
}

\section{App creation}{

The \code{create_app} method creates a new app registration. By default, a new app will have a randomly generated strong password with duration of 2 years. To skip assigning a password, set the \code{add_password} argument to FALSE.

The \code{certificate} argument allows authenticating via a certificate instead of a password. This should be a character string containing the certificate public key (aka the CER file). Alternatively it can be an list, or an object of class \code{AzureKeyVault::stored_cert} representing a certificate stored in an Azure Key Vault. See the examples below.

A new app will also have a service principal created for it by default. To disable this, set \code{create_service_principal=FALSE}.
}

\section{List methods}{

All \verb{list_*} methods have \code{filter} and \code{n} arguments to limit the number of results. The former should be an \href{https://learn.microsoft.com/en-us/graph/query-parameters#filter-parameter}{OData expression} as a string to filter the result set on. The latter should be a number setting the maximum number of (filtered) results to return. The default values are \code{filter=NULL} and \code{n=Inf}. If \code{n=NULL}, the \code{ms_graph_pager} iterator object is returned instead to allow manual iteration over the results.

Support in the underlying Graph API for OData queries is patchy. Not all endpoints that return lists of objects support filtering, and if they do, they may not allow all of the defined operators. If your filtering expression results in an error, you can carry out the operation without filtering and then filter the results on the client side.
}

\examples{
\dontrun{

# start a new Graph session
gr <- ms_graph$new(tenant="myaadtenant.onmicrosoft.com")

# authenticate with credentials in a file
gr <- ms_graph$new(config_file="creds.json")

# authenticate with device code
gr <- ms_graph$new(tenant="myaadtenant.onmicrosoft.com", app="app_id", auth_type="device_code")

# retrieve an app registration
gr$get_app(app_id="myappid")

# create a new app and associated service principal, set password duration to 10 years
app <- gr$create_app("mynewapp", password_duration=10)

# delete the app
gr$delete_app(app_id=app$properties$appId)
# ... but better to call the object's delete method directly
app$delete()

# create an app with authentication via a certificate
cert <- readLines("mycert.cer")
gr$create_app("mycertapp", password=FALSE, certificate=cert)

# retrieving your own user details (assuming interactive authentication)
gr$get_user()

# retrieving another user's details
gr$get_user("username@myaadtenant.onmicrosoft.com")
gr$get_user(email="firstname.lastname@mycompany.com")
gr$get_user(name="Hong Ooi")

# get an AAD object (a group)
id <- gr$get_user()$list_group_memberships()[1]
gr$get_aad_object(id)

# list the users in the tenant
gr$list_users()

# list (guest) users with a 'gmail.com' email address
gr$list_users(filter="endsWith(mail,'gmail.com')")

# list Microsoft 365 groups
gr$list_groups(filter="groupTypes/any(c:c eq 'Unified')")

}
}
\seealso{
\link{create_graph_login}, \link{get_graph_login}

\href{https://learn.microsoft.com/en-us/graph/overview}{Microsoft Graph overview},
\href{https://learn.microsoft.com/en-us/graph/api/overview?view=graph-rest-1.0}{REST API reference}
}
