## -----------------------------------------------------------------------------
#| label: install MSUthemes from CRAN
#| eval: false
#| include: true
# install.packages("MSUthemes")


## -----------------------------------------------------------------------------
#| label: install MSUthemes from GitHub repository
#| eval: false
#| include: true
# devtools::install_github("emilioxavier/MSUthemes", force=TRUE)


## -----------------------------------------------------------------------------
#| label: load MSUthemes
library(MSUthemes)


## -----------------------------------------------------------------------------
#| label: MSU Green barplot
library(dplyr)
barplot(C150_4 ~ entry_term, data=filter(BigTen, name=="MSU"),
        col = msu_green,
        main = "MSU Graduation Rate (1997 - 2023)",
        xlab = "Graduation Year of Incoming Class", ylab = "6-Year Graduation Rate")


## -----------------------------------------------------------------------------
#| label: MSU qualitative palette 1 (base R)
set_msu_palette("msu_qual1")
plot(1:13, 1:13, col=1:13, pch=19, cex=3, xlab="", ylab="",
     main = "Example of Michigan State University (MSU) Colour Palette Plot",
     sub = "Source: Sequential Values")
abline(h=1:13, v=1:13, col = "lightgrey")



## -----------------------------------------------------------------------------
#| label: MSU qualitative palette 1 (base R; barplot)
set_msu_par()
barplot(C150_4 ~ entry_term, data=filter(BigTen, name == "MSU"),
        col = set_msu_palette("msu_qual1"),
        main = "MSU Graduation Rate (1997 - 2023)",
        xlab = "Graduation Year of Incoming Class", ylab = "6-Year Graduation Rate")


## -----------------------------------------------------------------------------
#| label: MSU qualitative palette 1 (MSU theme in base R)
set_msu_par()
plot(1:13, 1:13, col=1:13, pch=19, cex=3, xlab="", ylab="",
     main = "Example of Michigan State University (MSU) Colour Palette Plot",
     sub = "Source: Sequential Values")
abline(h=1:13, v=1:13, col = "lightgrey")


## -----------------------------------------------------------------------------
#| label: reset graphics and extract MSU dataset
#| code-fold: false
library(ggplot2)
library(dplyr)

set_msu_par()  ## reset graphics par(ameters)

MSU_df <- filter(BigTen, name == "MSU")


## -----------------------------------------------------------------------------
#| label: MSU grellow barplot (ggplot2)
#| echo: true
#| message: false
#| warning: false
ggplot(data = MSU_df,
       mapping = aes(x = entry_term, y = UGDS)) +
  geom_col(fill = msu_yellow)


## -----------------------------------------------------------------------------
#| label: MSU qualitative palette 1 (ggplot2; barplot)
ggplot(data = slice(MSU_df, 1:13),
       mapping = aes(x = entry_term, y = UGDS, fill = factor(entry_term))) +
  geom_col() +
  scale_fill_msu_d(palette = "msu_qual1")


## -----------------------------------------------------------------------------
#| label: MSU qualitative palette 2 (ggplot; scatterplot)
ggplot(data = slice(MSU_df, 1:4),
       mapping = aes(x = entry_term, y = UGDS, colour = factor(entry_term))) +
  geom_point(size = 6) +
  scale_colour_msu_d(palette = "msu_qual2")


## -----------------------------------------------------------------------------
#| label: MSU green sequence (ggplot2; barplot)
ggplot(data = slice(MSU_df, 1:13),
       mapping = aes(x = entry_term, y = UGDS, fill = UGDS)) +
  geom_col() +
  scale_fill_msu_c(palette = "msu_seq")


## -----------------------------------------------------------------------------
#| label: MSU green sequence (ggplot2; scatterplot)
ggplot(data = MSU_df,
       mapping = aes(x = entry_term, y = UGDS, colour = UGDS)) +
  geom_point(size = 6) +
  scale_colour_msu_c(palette = "msu_seq")


## -----------------------------------------------------------------------------
#| label: MSU divergent sequence (red-to-blue; ggplot2; barplot)
ggplot(data = MSU_df,
       mapping = aes(x = entry_term, y = UGDS, fill = UGDS)) +
  geom_col() +
  scale_fill_msu_c(palette = "msu_div")


## -----------------------------------------------------------------------------
#| label: MSU divergent sequence (orange-to-purple; ggplot2; barplot)
ggplot(data = MSU_df,
       mapping = aes(x = entry_term, y = UGDS, fill = UGDS)) +
  geom_col() +
  scale_fill_gradient2(low = msu_orange,
                       high = msu_purple,
                       midpoint = median(MSU_df$UGDS, na.rm = TRUE))


## -----------------------------------------------------------------------------
#| label: MSU divergent sequence (red-to-blue; ggplot2+theme_MSU; barplot)
BigTen_2023 <- filter(BigTen, entry_term == 2023)

ggplot(data = BigTen_2023, aes(x = reorder(name, UGDS), y = UGDS, fill = ADM_RATE)) +
  geom_col() +
  scale_fill_msu_c(palette = "msu_div") +
  scale_y_continuous(labels = scales::comma) +
  labs(x = NULL,
       y = "Undergraduate Enrollment",
       title = "Big Ten Enrollment (2023)",
       subtitle = "Bars Shaded by Admission Rate",
       fill = "Admission Rate") +
  theme_MSU() +
  theme(axis.text.x = element_text(angle = 45, hjust = 1))


## -----------------------------------------------------------------------------
#| label: Big Ten colour palette (ggplot2+theme_MSU; scatterplot)
BigTen_2023 <- filter(BigTen, entry_term == 2023)

ggplot(data = BigTen_2023,
       aes(x = ADM_RATE, y = C150_4, size = UGDS, colour = name, label = name)) +
  geom_point(alpha = 0.7) +
  ggrepel::geom_text_repel(size = 3,
                           colour = "black",
                           show.legend = FALSE, max.overlaps = 20) +
  scale_x_continuous(labels = scales::percent) +
  scale_y_continuous(labels = scales::percent) +
  scale_colour_manual(values = bigten_colors_primary) +
  scale_size_continuous(labels = scales::comma, range = c(3, 12)) +
  guides(size = guide_legend(title = "Undergraduate\nEnrollment"),
         colour = "none") +
  labs(x = "Admission Rate",
       y = "6-Year Graduation Rate",
       title = "Big Ten Institutional Comparison (2023)",
       subtitle = "Each institution shown in its official primary color") +
  theme_MSU()


## -----------------------------------------------------------------------------
#| label: export examples (base R)
#| eval: false
# pdf(file = "MSU-feature-plot.pdf",  # name of file
#     width = 7,                      # width in inches
#     height = 5                      # height in inches
#     )
# barplot(C150_4 ~ entry_term, data = filter(BigTen, name == "MSU"),
#         col = msu_green,
#         main = "MSU Graduation Rate (1997 - 2023)",
#         xlab = "Graduation Year of Incoming Class",
#         ylab = "6-Year Graduation Rate")
# dev.off()


## -----------------------------------------------------------------------------
#| label: export examples (ggplot2 + ggsave)
#| eval: false
# # Create a plot
# p <- ggplot(data = MSU_df, aes(x = entry_term, y = UGDS, fill = UGDS)) +
#   geom_col() +
#   scale_fill_msu_c(palette = "msu_seq") +
#   labs(title = "MSU Undergraduate Enrollment",
#        x = "Entry Term",
#        y = "Undergraduate Enrollment") +
#   theme_MSU()
# 
# # Save the plot
# ggsave(filename = "MSU-enrollment.pdf",
#        plot = p,
#        width = 8,
#        height = 6,
#        dpi = 300)
# 
# # For PNG format
# ggsave(filename = "MSU-enrollment.png",
#        plot = p,
#        width = 8,
#        height = 6,
#        dpi = 300)

