package slot;
use strict;
use warnings;
our $VERSION = '0.03';
require XSLoader;
XSLoader::load('slot', $VERSION);
1;

__END__

=head1 NAME

slot - global reactive state slots with optional watchers

=head1 SYNOPSIS

    # Define and use slots
    package Config;
    use slot qw(app_name debug);
    
    app_name("MyApp");
    debug(1);
    
    # Access from another package (same underlying storage)
    package Service;
    use slot qw(app_name);
    
    print app_name();  # "MyApp"
    app_name("Changed");

    # Watchers (reactive)
    slot::watch('app_name', sub {
        my ($name, $value) = @_;
        print "app_name changed to: $value\n";
    });
    
    slot::unwatch('app_name');  # Remove all watchers

=head1 DESCRIPTION

C<slot> provides fast, globally shared named storage slots.
Slots are shared across all packages - importing the same slot name in different
packages gives access to the same underlying value.

Key features:

=over 4

=item * B<Fast> - Custom ops

=item * B<Global> - Slots are shared across packages by name

=item * B<Reactive> - Optional watchers fire on value changes

=item * B<Lazy watchers> - No overhead unless you use C<watch()>

=back

=head1 FUNCTIONS

=head2 import

    use slot qw(foo bar baz);

Imports slot accessors into the calling package. Each accessor is both a
getter and setter:

    foo();       # get
    foo(42);     # set and returns value

=head2 slot::get

    my $val = slot::get('name');

Get a slot value by name (without importing an accessor).

=head2 slot::set

    slot::set('name', $value);

Set a slot value by name.

=head2 slot::watch

    slot::watch('name', sub { my ($name, $val) = @_; ... });

Register a callback that fires whenever the slot value changes.

=head2 slot::unwatch

    slot::unwatch('name');            # Remove all watchers
    slot::unwatch('name', $coderef);  # Remove specific watcher

=head2 slot::slots

    my @names = slot::slots();

Returns a list of all defined slot names.

=head1 THREAD SAFETY

For thread-safe data sharing, store C<threads::shared> variables in slots:

    use threads;
    use threads::shared;
    use slot qw(config);

    # Create shared data and store in slot
    my %shared :shared;
    $shared{counter} = 0;
    config(\%shared);

    # Threads can now safely share data via the slot
    my @threads = map {
        threads->create(sub {
            my $cfg = config();
            lock(%$cfg);
            $cfg->{counter}++;
        });
    } 1..10;

    $_->join for @threads;
    print config()->{counter};  # 10

The slot provides the global accessor; C<threads::shared> provides the
thread-safe storage.

=head1 FORK BEHAVIOR

After C<fork()>, child processes get a copy of slot values (copy-on-write).
Changes in child processes do not affect the parent, and vice versa.

=head1 BENCHMARK

bench.pl is included

	=== SETTER BENCHMARK ===

			  Rate pp_closure    pp_hash   raw_hash       slot
	pp_closure  12277720/s         --       -14%       -82%       -96%
	pp_hash     14298759/s        16%         --       -80%       -96%
	raw_hash    70063941/s       471%       390%         --       -79%
	slot       340183182/s      2671%      2279%       386%         --

	=== GETTER BENCHMARK ===

			 Rate pp_closure    pp_hash   raw_hash       slot
	pp_closure 11513773/s         --       -14%       -79%       -84%
	pp_hash    13345509/s        16%         --       -76%       -82%
	raw_hash   54947210/s       377%       312%         --       -25%
	slot       72793704/s       532%       445%        32%         --


=head1 AUTHOR

LNATION <email@lnation.org>

=head1 LICENSE

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
