#!/usr/local/bin/perl

#  this script tests the expression evaluator.  it includes some
#  examples of how to use various features.

require  "expr_eval.pm";


#  initialize

$tests = getTests();

$report = "";
$errorReport = "";

$testCount = 0;
$errorCount = 0;

$symTable1
   = {
   	"SYM1"	=>	[$Math::ExprEval::exprValueStringType,"symbol table 1"],
     };

$symTable2
   = {
   	"SYM2"	=>	[$Math::ExprEval::exprValueStringType,"symbol table 2"],
     };

$symbolTableList = [$symTable1, $symTable2];



#  loop through tests

foreach $test (split(/<EOT[^>\n]*>\s*/,$tests))
{
  skip if ($test =~ /^\s*$/);


#  get the expression, comments, options and expected results for the test

  $testCount++;

  $test = "\n" . $test;
  $test =~ /\nExpression\s*(\([^)]*\)\s*|):(.*)\n([^\000]*)/i;
  $comments = $`;
  $options = $1;
  $expression = $2;
  $expectedResults = $3;
  $options =~ s/^\(//g;
  $options =~ s/\)\s*//g;

  $comments =~ s/^\s*//;
  $comments =~ s/\s*$//;

  $expression =~ s/^\s*//;
  $expression =~ s/\s*$//;

  $expectedResults =~ s/^\s*//;
  $expectedResults =~ s/\s*$//;
  $expectedResults .= "\n";


#  create the arguement list for the call to the expression evaluator.
#  arguements are included depending on the options specified.

  @args = ('symbol-table-list' => $symbolTableList);

  push(@args, 'save-token-list' => "t" )
    if($options =~ /(save-tokens|replay-tokens)/i);
  push(@args, 'token-function' => \&tokenFunction )
    if($options =~ /token-function/i);
  push(@args, 'operator-function' => \&operatorFunction )
    if($options =~ /operator-function/i);
  push(@args, 'function-handler' => \&functionHandler )
    if($options =~ /function-handler/i);
  push(@args, 'syntax-check-only' => "t" )
    if($options =~ /syntax-check-only/i);
  push(@args, 'boolean-keywords' => "t" )
    if($options =~ /boolean-keywords/i);


#  call the expression evaluator function.  if the 'replay-tokens'
#  option is specified, the function is called a second time using
#  the token list generated by the first call.

  ($returnType, $returnValue, $returnErrorPosition,
              $returnErrorMessage, $returnTokenList,
	      $returnDeveloperErrorMessage)
      = Math::ExprEval::evalExpression($expression, @args);

  if(($options =~ /replay-tokens/i) && (defined $returnTokenList))
  {
    push(@args, 'token-list' => $returnTokenList);

    ($returnType, $returnValue, $returnErrorPosition,
                $returnErrorMessage, $dummy,
	        $returnDeveloperErrorMessage)
        = Math::ExprEval::evalExpression($expression, @args);
  }


#  if a token list was returned, create a string listing each token.

  if( ! defined $returnTokenList)
  {
    $tokenList = "Token list:\t<NONE>\n";
  }
  else
  {
    $tokenList = "Token list:\n";

    foreach $tokenInfo (@$returnTokenList)
    {
      $tokenList
        .= "    ($$tokenInfo[0], '$$tokenInfo[1]', $$tokenInfo[2], "
           . (( ! defined $$tokenInfo[3])
	        ? "<NONE>"
	        : ((ref($$tokenInfo[3]) eq "")
                    ? $$tokenInfo[3]
	            : ( "<" . ref($$tokenInfo[3]) . ">" )))
          . ")\n";
    }

    $tokenList .= "End of token list\n";
  }


#  create a string listing the resulting information, including any
#  error messages, from the call to the expression evaluator.

  $results
    = "Return type:\t"
       . (( ! defined $returnType)
	    ? "<NONE>"
	    : $returnType)
       . "\n"
    . "Return value:\t"
       . (( ! defined $returnValue)
	    ? "<NONE>"
	    : ((ref($returnValue) eq "")
                ? $returnValue
	        : ( "<" . ref($returnValue) . ">" )))
       . "\n"
    . "Error position:\t"
       . ((defined $returnErrorPosition)
           ? $returnErrorPosition
	   : "<NONE>" )
       . "\n"
    . "Error message:\t"
       . ((defined $returnErrorMessage)
           ? $returnErrorMessage
	   : "<NONE>" )
       . "\n"
    . "Dev message:\t"
       . (( ! defined $returnDeveloperErrorMessage)
	   ? "<NONE>\n" 
           : (($returnDeveloperErrorMessage !~ /\n/)
	      ? "$returnDeveloperErrorMessage\n"
	      : "<->\n$returnDeveloperErrorMessage"))
    . $tokenList;


#  compare the actual results to the expected results.  if they are
#  different, add an entry to the error report.

  if($results ne $expectedResults)
  {
    $errorCount++;

    $errorReport .=
       "Test:\t\t$testCount\n"
       . "Error number:\t$errorCount\n\n"
       . "Expression:\t$expression\n"
       . "Options:\t" . (($options eq "") ? "<none>" : ("'" . $options . "'")) . "\n\n"
       . (($comments eq "") ? "\n" : ($comments . "\n\n\n"))
       . "Actual results:\n\n"
       . $results . "\n\n"
       . "Expected results:\n\n"
       . $expectedResults . "\n\n"
       . "----------------------------\n\n";
  }


#  add an entry about the results to the standard report

  $report .=
     "Test:\t\t$testCount\n\n"
     . "Expression:\t$expression\n"
     . "Options:\t" . (($options eq "") ? "<none>" : ("'" . $options . "'")) . "\n\n"
     . (($comments eq "") ? "" : ($comments . "\n\n\n"))
     . $results . "\n\n"
     . "----------------------------\n\n";
}



#  print out final error and general reports

if($errorReport eq "")
{
  $errorReport = "NO ERRORS FOUND FOUND TESTING EXPRESSION EVALUATION\n";
}
else
{
  $errorReport
    =  "Total test count:\t    $testCount\n"
         . "Total test error count:\t    $errorCount\n\n"
         . "----------------------------\n\n"
         . $errorReport
         . "END OF ERROR REPORT\n";
}

$report
   = "Total test count:\t$testCount\n\n"
     . "----------------------------\n\n"
     . $report
     . "END OF REPORT\n";



print "\n" . $errorReport . "\n\n============================================================================\n\n\n"
         . $report;



#  ----------------  VARIOUS TEST USER FUNCTIONS   ------------------------


sub tokenFunction
{
  my $arg = shift;
  my $checkSyntaxOnly = shift;
  my $string = shift;

  my $returnTokenString = undef;
  my $returnTokenType = undef;
  my $returnTokenValue = undef;
  my $x;

  if($string =~ /^(%test-token%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::tokenTypeString;
    $returnTokenValue = "string from token function";
  }
  elsif($string =~ /^(%error-message%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::tokenTypeError;
    $returnTokenValue = "ERROR MESSAGE";
  }
  elsif($string =~ /^(%execution-error%)/i)
  {
    $x = 1;
    $x/0;
  }
  elsif($string =~ /^(%unary-operator%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::tokenTypeUnaryOperator;
    $returnTokenValue = $returnTokenString;
  }
  elsif($string =~ /^(%unary-operator-error%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::tokenTypeUnaryOperator;
    $returnTokenValue = $returnTokenString;
  }
  elsif($string =~ /^(%binary-operator%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::minTokenTypeUserBinaryOperator;
              #  the token type for a binary operator is its precedence
    $returnTokenValue = $returnTokenString;
  }
  elsif($string =~ /^(%binary-operator-error%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::minTokenTypeUserBinaryOperator;
              #  the token type for a binary operator is its precedence
    $returnTokenValue = $returnTokenString;
  }
  elsif($string =~ /^(%ignore_token%)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::tokenTypeIgnore;
    $returnTokenValue = "'token should be ignored'";
  }
  elsif($string =~ /^(my_special_data)/i)
  {
    $returnTokenString = $1;
    $returnTokenType = $Math::ExprEval::tokenTypeExprValue;
    $returnTokenValue
      = [ $Math::ExprEval::minExprUserAbstractType + 1 , "some-special-info" ]
  }

  return ($returnTokenString, $returnTokenType, $returnTokenValue);
}



sub operatorFunction
{
  my $arg = shift;
  my $checkSyntaxOnly = shift;
  my $operatorTokenString = shift;
  my $operatorTokenType = shift;
  my $operatorTokenValue = shift;
  my $operandType = shift;
  my $operandValue = shift;
  my $operand2Type = shift;
  my $operand2Value = shift;

  my $returnType = undef;
  my $returnValue = undef;

  if($operatorTokenString eq "%unary-operator%")
  {
    if($operandType == $Math::ExprEval::exprValueStringType)
    {
      $returnType = $Math::ExprEval::exprValueStringType;
      $returnValue = "string from unary operation";
    }
    else
    {
      $returnType = $Math::ExprEval::exprValueErrorType;
      $returnValue = "string expected after operator";
    }
  }
  elsif($operatorTokenString eq "%unary-operator-error%")
  {
    $x = 1;
    $x/0;
  }
  elsif($operatorTokenString eq "%binary-operator%")
  {
    if($operandType != $Math::ExprEval::exprValueStringType)
    {
      $returnType = $Math::ExprEval::exprValueErrorType;
      $returnValue = "string expected before operator";
    }
    elsif($operand2Type != $Math::ExprEval::exprValueStringType)
    {
      $returnType = $Math::ExprEval::exprValueErrorType;
      $returnValue = "string expected after operator";
    }
    else
    {
      $returnType = $Math::ExprEval::exprValueStringType;
      $returnValue = "($operand2Value,$operandValue)"
    }
  }
  elsif($operatorTokenString eq "%binary-operator-error%")
  {
    $x = 1;
    $x/0;
  }

  return ($returnType, $returnValue);
}



sub functionHandler
{
  my $arg = shift;
  my $checkSyntaxOnly = shift;
  my $functionIdentifier = shift;

  my $returnType = undef;
  my $returnValue = undef;

  $functionIdenifier = "\U$functionIdenifier";

  if($functionIdentifier eq "CONCAT")
  {
    $returnValue = "";

    foreach $arg (@_)
    {
      if(defined $returnType)
      {
      }
      elsif($$arg[0] != $Math::ExprEval::exprValueBoolType)
      {
        $returnValue .= $$arg[1] . ", ";
      }
      else
      {
        $returnType = $Math::ExprEval::exprValueErrorType;
        $returnValue = "boolean arguement found";
      }
    }

    if( ! defined $returnType)
    {
      $returnType = $Math::ExprEval::exprValueStringType;
      $returnValue =~ s/, *$//;
    }
  }
  elsif($functionIdentifier eq "DIVBYZERO")
  {
    $x = 1;
    $x/0;
  }

  return ($returnType, $returnValue);
}


# -----------------    INFORMATION FOR TESTS    ---------------------------

sub getTests
{
return <<'EOF';

check for unknown character

Expression:     ~

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	unexpected character
Dev message:	character = '~'
Token list:	<NONE>

<EOT>

check for quoted strings

Expression (save-tokens):  'hello world, ' + "hello workd 2, " + `hello world 3`

Return type:	2
Return value:	hello world, hello workd 2, hello world 3
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, ''hello world, '', 3, hello world, )
    (16, '+', 24, +)
    (18, '"hello workd 2, "', 3, hello workd 2, )
    (36, '+', 24, +)
    (38, '`hello world 3`', 3, hello world 3)
    (53, '', 1, )
    (53, '', 1, )
End of token list

<EOT>

check for string with missing quote

Expression:     'hello world, ' + "missing quote

Return type:	<NONE>
Return value:	<NONE>
Error position:	18
Error message:	missing close quote
Dev message:	quote = /"/
Token list:	<NONE>

<EOT>

check for token function ignoring token

Expression (save-tokens,token-function):  %ignore_token%  "HERE"  %ignore_token%

Return type:	2
Return value:	HERE
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '%ignore_token%', 14, 'token should be ignored')
    (16, '"HERE"', 3, HERE)
    (24, '%ignore_token%', 14, 'token should be ignored')
    (38, '', 1, )
    (38, '', 1, )
End of token list

<EOT>

check lexical scanning for numbers

Expression (save-tokens):  55 + 23.0 + .25 + 0.5 + 9.25

Return type:	1
Return value:	88
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '55', 4, 55)
    (3, '+', 24, +)
    (5, '23.0', 4, 23)
    (10, '+', 24, +)
    (12, '.25', 4, 0.25)
    (16, '+', 24, +)
    (18, '0.5', 4, 0.5)
    (22, '+', 24, +)
    (24, '9.25', 4, 9.25)
    (28, '', 1, )
    (28, '', 1, )
End of token list

<EOT>

check lexical scanning for numbers with scientific notation

Expression (save-tokens):  1000E-2 + 2.0E+1 + .25E2 + 0.5E+1 + 9.25E+1

Return type:	1
Return value:	152.5
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1000E-2', 4, 10)
    (8, '+', 24, +)
    (10, '2.0E+1', 4, 20)
    (17, '+', 24, +)
    (19, '.25E2', 4, 25)
    (25, '+', 24, +)
    (27, '0.5E+1', 4, 5)
    (34, '+', 24, +)
    (36, '9.25E+1', 4, 92.5)
    (43, '', 1, )
    (43, '', 1, )
End of token list

<EOT>

check lexical scanning for malformed number

Expression:  0.5.

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	incorrectly specified number
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check for 'TRUE' boolean constant

Expression (save-tokens):  true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (4, '', 1, )
    (4, '', 1, )
End of token list

<EOT>

check for 'FALSE' boolean constant

Expression (save-tokens):  FALSE

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'FALSE', 5, <ARRAY>)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check lexical scanning for malformed number

Expression:  6W

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	incorrectly specified number
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check lexical scanning for malformed number

Expression:  .7896W

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	incorrectly specified number
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check user token function to return a standard string data type

Expression (token-function,save-tokens):  %test-token%

Return type:	2
Return value:	string from token function
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '%test-token%', 3, string from token function)
    (12, '', 1, )
    (12, '', 1, )
End of token list

<EOT>

check user token function for returning a token for a user data type

Expression (token-function,save-tokens):  my_special_data

Return type:	11
Return value:	some-special-info
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'my_special_data', 5, <ARRAY>)
    (15, '', 1, )
    (15, '', 1, )
End of token list

<EOT>

check user token function error return

Expression (token-function):  %error-message%

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	ERROR MESSAGE
Dev message:	function is code
Token list:	<NONE>

<EOT>

check user token function execution error
(note:  format of error message may vary depending on perl version, system, etc)

Expression (token-function):  %execution-error%

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	system error
Dev message:	<->
function is code
Illegal division by zero at test.pl line 256.
Token list:	<NONE>

<EOT>

check retrieving symbols from symbol tables

Expression (save-tokens):  sym1 + ", " + sym2

Return type:	2
Return value:	symbol table 1, symbol table 2
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'sym1', 2, sym1)
    (5, '+', 24, +)
    (7, '", "', 3, , )
    (12, '+', 24, +)
    (14, 'sym2', 2, sym2)
    (18, '', 1, )
    (18, '', 1, )
End of token list

<EOT>

check for unknown symbol

Expression:  "XXX" + sym3

Return type:	<NONE>
Return value:	<NONE>
Error position:	11
Error message:	unrecognized identifier
Dev message:	<->
Token information:
    Position:	8
    String:	sym3
    Type:	2
    Value:	sym3
Token list:	<NONE>

<EOT>

check for unary negative

Expression (save-tokens):      5 + -3

Return type:	1
Return value:	2
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '5', 4, 5)
    (2, '+', 24, +)
    (4, '-', 24, -)
    (5, '3', 4, 3)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check for double unary negative

Expression (save-tokens):      - -3

Return type:	1
Return value:	3
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '-', 24, -)
    (2, '-', 24, -)
    (3, '3', 4, 3)
    (4, '', 1, )
    (4, '', 1, )
End of token list

<EOT>

check for unary negative acting on non-numeric

Expression:      5 + -"XXX"

Return type:	<NONE>
Return value:	<NONE>
Error position:	4
Error message:	numeric value expected next
Dev message:	<->
Value information:
    Type:	2  (string)
    Value:	XXX
Token list:	<NONE>

<EOT>

check for NOT on true

Expression (save-tokens):      ! true

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '!', 13, !)
    (2, 'true', 5, <ARRAY>)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check for NOT on false

Expression (save-tokens):      ! false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '!', 13, !)
    (2, 'false', 5, <ARRAY>)
    (7, '', 1, )
    (7, '', 1, )
End of token list

<EOT>

check for double NOT

Expression (save-tokens):      ! ! true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '!', 13, !)
    (2, '!', 13, !)
    (4, 'true', 5, <ARRAY>)
    (8, '', 1, )
    (8, '', 1, )
End of token list

<EOT>

check for NOT acting on non-boolean

Expression:      ! "XXX"

Return type:	<NONE>
Return value:	<NONE>
Error position:	0
Error message:	boolean value expected next
Dev message:	<->
Value information:
    Type:	2  (string)
    Value:	XXX
Token list:	<NONE>

<EOT>

check for special unary operator without operator function

Expression (token-function):      %unary-operator% "XXX"

Return type:	<NONE>
Return value:	<NONE>
Error position:	15
Error message:	operator not evaluated by function
Dev message:	<->
No function specified.
Token information:
    Position:	0
    String:	%unary-operator%
    Type:	13
    Value:	%unary-operator%
Token list:	<NONE>

<EOT>

check for special unary operator

Expression (token-function,operator-function,save-tokens):      %unary-operator% "XXX"

Return type:	2
Return value:	string from unary operation
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '%unary-operator%', 13, %unary-operator%)
    (17, '"XXX"', 3, XXX)
    (22, '', 1, )
    (22, '', 1, )
End of token list

<EOT>

check for special unary operator acting on non-string

Expression (token-function,operator-function):      %unary-operator% 100

Return type:	<NONE>
Return value:	<NONE>
Error position:	15
Error message:	string expected after operator
Dev message:	<->
function is code
Token information:
    Position:	0
    String:	%unary-operator%
    Type:	13
    Value:	%unary-operator%
Token list:	<NONE>

<EOT>

check for operator function error
(note:  format of error message may vary depending on perl version, system, etc)

Expression (token-function,operator-function):      %unary-operator-error% "XXX"

Return type:	<NONE>
Return value:	<NONE>
Error position:	21
Error message:	system error
Dev message:	<->
function is code
Token information:
    Position:	0
    String:	%unary-operator-error%
    Type:	13
    Value:	%unary-operator-error%
Illegal division by zero at test.pl line 334.
Token list:	<NONE>

<EOT>

check for parentheses

Expression (save-tokens):      ( 5 + 6 )

Return type:	1
Return value:	11
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '(', 6, ()
    (2, '5', 4, 5)
    (4, '+', 24, +)
    (6, '6', 4, 6)
    (8, ')', 7, ))
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check for missing close parenthesis

Expression:      ( 5 + 6

Return type:	<NONE>
Return value:	<NONE>
Error position:	7
Error message:	unexpected end found
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check for missing close parenthesis

Expression:      ( 5 + 6 ,

Return type:	<NONE>
Return value:	<NONE>
Error position:	8
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	8
    String:	,
    Type:	8
    Value:	,
Token list:	<NONE>

<EOT>

check for missing closing function parenthesis

Expression (function-handler):      CONCAT("555", 8, 7, "9"

Return type:	<NONE>
Return value:	<NONE>
Error position:	22
Error message:	unexpected end found
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check for empty arguement

Expression (function-handler):      CONCAT( , "555", 8, 7, "9")

Return type:	<NONE>
Return value:	<NONE>
Error position:	8
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	8
    String:	,
    Type:	8
    Value:	,
Token list:	<NONE>

<EOT>

check for no processing function

Expression:      CONCAT("555", 8, 7, "9")

Return type:	<NONE>
Return value:	<NONE>
Error position:	23
Error message:	no function declared for handling function calls
Dev message:	function identifier = 'CONCAT'
Token list:	<NONE>

<EOT>

check for function call

Expression (function-handler,save-tokens):      CONCAT("555", 8, 7, "9")

Return type:	2
Return value:	555, 8, 7, 9
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'CONCAT', 12, CONCAT)
    (6, '(', 6, ()
    (7, '"555"', 3, 555)
    (12, ',', 8, ,)
    (14, '8', 4, 8)
    (15, ',', 8, ,)
    (17, '7', 4, 7)
    (18, ',', 8, ,)
    (20, '"9"', 3, 9)
    (23, ')', 7, ))
    (24, '', 1, )
    (24, '', 1, )
End of token list

<EOT>

check for function detecting arguement error

Expression (function-handler):      CONCAT("555", true, 7, "9")

Return type:	<NONE>
Return value:	<NONE>
Error position:	26
Error message:	boolean arguement found
Dev message:	<->
function identifier = 'CONCAT'
function is code
Token list:	<NONE>

<EOT>

check for function being unknown

Expression (function-handler):      UNKNOWN("555", true, 7, "9")

Return type:	<NONE>
Return value:	<NONE>
Error position:	27
Error message:	function may be unknown (no value returned for function call)
Dev message:	<->
function identifier = 'UNKNOWN'
function is code
Token list:	<NONE>

<EOT>

check for ternary function without boolean

Expression:      5 ? 5 : 5

Return type:	<NONE>
Return value:	<NONE>
Error position:	2
Error message:	boolean value expected before
Dev message:	<->
Token information:
    Position:	2
    String:	?
    Type:	9
    Value:	?
Token list:	<NONE>

<EOT>

check for ternary function without colon

Expression:      true ? 5 

Return type:	<NONE>
Return value:	<NONE>
Error position:	<NONE>
Error message:	unexpected end found
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check for ternary function without colon

Expression:      true ? 5 , 5

Return type:	<NONE>
Return value:	<NONE>
Error position:	9
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	9
    String:	,
    Type:	8
    Value:	,
Token list:	<NONE>

<EOT>

check for ternary function with true boolean

Expression (save-tokens):      true ? 5 : 6

Return type:	1
Return value:	5
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '?', 9, ?)
    (7, '5', 4, 5)
    (9, ':', 10, :)
    (11, '6', 4, 6)
    (12, '', 1, )
    (12, '', 1, )
    (12, '', 1, )
End of token list

<EOT>

check for ternary function with false boolean

Expression (save-tokens):      false ? 5 : 6

Return type:	1
Return value:	6
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '?', 9, ?)
    (8, '5', 4, 5)
    (10, ':', 10, :)
    (12, '6', 4, 6)
    (13, '', 1, )
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check for execution error while handling function call
(note:  format of error message may vary depending on perl version, system, etc)

Expression (function-handler):      DIVBYZERO("555", true, 7, "9")

Return type:	<NONE>
Return value:	<NONE>
Error position:	29
Error message:	system error
Dev message:	<->
function identifier = 'DIVBYZERO'
function is code
Illegal division by zero at test.pl line 405.
Token list:	<NONE>

<EOT>

check for unexpected element

Expression:      5 "XXX"

Return type:	<NONE>
Return value:	<NONE>
Error position:	6
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	2
    String:	"XXX"
    Type:	3
    Value:	XXX
Token list:	<NONE>

<EOT>

check for unexpected element

Expression:      5 ,

Return type:	<NONE>
Return value:	<NONE>
Error position:	2
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	2
    String:	,
    Type:	8
    Value:	,
Token list:	<NONE>

<EOT>

check for unexpected end

Expression:      5 +

Return type:	<NONE>
Return value:	<NONE>
Error position:	3
Error message:	unexpected end found
Dev message:	<NONE>
Token list:	<NONE>

<EOT>

check for unexpected element

Expression:      5 + ||

Return type:	<NONE>
Return value:	<NONE>
Error position:	5
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	4
    String:	||
    Type:	21
    Value:	||
Token list:	<NONE>

<EOT>


check for unexpected element

Expression:      5 )

Return type:	<NONE>
Return value:	<NONE>
Error position:	2
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	2
    String:	)
    Type:	7
    Value:	)
Token list:	<NONE>

<EOT>

check for unexpected element

Expression:      8 + 3(

Return type:	<NONE>
Return value:	<NONE>
Error position:	5
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	5
    String:	(
    Type:	6
    Value:	(
Token list:	<NONE>

<EOT>

check for string concatenation

Expression (save-tokens):      "Hello, " + "World"

Return type:	2
Return value:	Hello, World
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"Hello, "', 3, Hello, )
    (10, '+', 24, +)
    (12, '"World"', 3, World)
    (19, '', 1, )
    (19, '', 1, )
End of token list

<EOT>

check for concatenating string followed by number

Expression (save-tokens):      "Hello, " + 53.25

Return type:	2
Return value:	Hello, 53.25
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"Hello, "', 3, Hello, )
    (10, '+', 24, +)
    (12, '53.25', 4, 53.25)
    (17, '', 1, )
    (17, '', 1, )
End of token list

<EOT>

check for trying to concatenate string followed by non-string/non-number

Expression:      "Hello, " + true

Return type:	<NONE>
Return value:	<NONE>
Error position:	10
Error message:	string or number expected after operator
Dev message:	<->
Token information:
    Position:	10
    String:	+
    Type:	24
    Value:	+
Value information:
    Type:	3  (bool)
    Value:	t
Token list:	<NONE>

<EOT>

check for trying to concatenate number followed by string

Expression:      32 + "world"

Return type:	<NONE>
Return value:	<NONE>
Error position:	3
Error message:	number expected after operator
Dev message:	<->
Token information:
    Position:	3
    String:	+
    Type:	24
    Value:	+
Value information:
    Type:	2  (string)
    Value:	world
Token list:	<NONE>

<EOT>

check for trying to concatenate non-string/non-number followed by string

Expression:      true + "world"

Return type:	<NONE>
Return value:	<NONE>
Error position:	5
Error message:	string or number expected before operator
Dev message:	<->
Token information:
    Position:	5
    String:	+
    Type:	24
    Value:	+
Value information:
    Type:	3  (bool)
    Value:	t
Token list:	<NONE>

<EOT>

check precedence handling so multiplication is followed by concatentation

Expression (save-tokens):      "Hello, " + 25 * 5

Return type:	2
Return value:	Hello, 125
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"Hello, "', 3, Hello, )
    (10, '+', 24, +)
    (12, '25', 4, 25)
    (15, '*', 25, *)
    (17, '5', 4, 5)
    (18, '', 1, )
    (18, '', 1, )
End of token list

<EOT>

check subtraction working on number and boolean

Expression:      5 - true

Return type:	<NONE>
Return value:	<NONE>
Error position:	2
Error message:	number expected after operator
Dev message:	<->
Token information:
    Position:	2
    String:	-
    Type:	24
    Value:	-
Value information:
    Type:	3  (bool)
    Value:	t
Token list:	<NONE>

<EOT>

check subtraction working on boolean and number

Expression:      true - 5

Return type:	<NONE>
Return value:	<NONE>
Error position:	5
Error message:	number expected before operator
Dev message:	<->
Token information:
    Position:	5
    String:	-
    Type:	24
    Value:	-
Value information:
    Type:	3  (bool)
    Value:	t
Token list:	<NONE>

<EOT>

check addition

Expression (save-tokens):      34 + 49

Return type:	1
Return value:	83
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '34', 4, 34)
    (3, '+', 24, +)
    (5, '49', 4, 49)
    (7, '', 1, )
    (7, '', 1, )
End of token list

<EOT>

check subtraction

Expression (save-tokens):      56 - 93

Return type:	1
Return value:	-37
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '56', 4, 56)
    (3, '-', 24, -)
    (5, '93', 4, 93)
    (7, '', 1, )
    (7, '', 1, )
End of token list

<EOT>

check multiplication

Expression (save-tokens):      5 * -7

Return type:	1
Return value:	-35
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '5', 4, 5)
    (2, '*', 25, *)
    (4, '-', 24, -)
    (5, '7', 4, 7)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check division

Expression (save-tokens):      42 / -6

Return type:	1
Return value:	-7
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '42', 4, 42)
    (3, '/', 25, /)
    (5, '-', 24, -)
    (6, '6', 4, 6)
    (7, '', 1, )
    (7, '', 1, )
End of token list

<EOT>

check division by zero
(note:  format of error message may vary depending on perl version, system, etc)

Expression:      45 / 0

Return type:	<NONE>
Return value:	<NONE>
Error position:	3
Error message:	error occurred when executing '/'
Dev message:	<->
Illegal division by zero at expr_eval.pm line 1584.
Token list:	<NONE>

<EOT>

check modulo

Expression (save-tokens):      34 % 5

Return type:	1
Return value:	4
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '34', 4, 34)
    (3, '%', 25, %)
    (5, '5', 4, 5)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check modulo by zero
(note:  format of error message may vary depending on perl version, system, etc)

Expression:      45 % 0

Return type:	<NONE>
Return value:	<NONE>
Error position:	3
Error message:	error occurred when executing '%'
Dev message:	<->
Illegal modulus zero at expr_eval.pm line 1598.
Token list:	<NONE>

<EOT>

check operation priorities

Expression (save-tokens):      25 + 6 * 8 - 64 / 8 + 4

Return type:	1
Return value:	69
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '25', 4, 25)
    (3, '+', 24, +)
    (5, '6', 4, 6)
    (7, '*', 25, *)
    (9, '8', 4, 8)
    (11, '-', 24, -)
    (13, '64', 4, 64)
    (16, '/', 25, /)
    (18, '8', 4, 8)
    (20, '+', 24, +)
    (22, '4', 4, 4)
    (23, '', 1, )
    (23, '', 1, )
End of token list

<EOT>

check operation priorities

Expression (save-tokens):      25 * 6 + 36 / 12 - 8 % 3

Return type:	1
Return value:	151
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '25', 4, 25)
    (3, '*', 25, *)
    (5, '6', 4, 6)
    (7, '+', 24, +)
    (9, '36', 4, 36)
    (12, '/', 25, /)
    (14, '12', 4, 12)
    (17, '-', 24, -)
    (19, '8', 4, 8)
    (21, '%', 25, %)
    (23, '3', 4, 3)
    (24, '', 1, )
    (24, '', 1, )
End of token list

<EOT>

check operation priorities being changed by parentheses

Expression (save-tokens):      (25 + 6) * (8 - 56) / (8 + 4)

Return type:	1
Return value:	-124
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '(', 6, ()
    (1, '25', 4, 25)
    (4, '+', 24, +)
    (6, '6', 4, 6)
    (7, ')', 7, ))
    (9, '*', 25, *)
    (11, '(', 6, ()
    (12, '8', 4, 8)
    (14, '-', 24, -)
    (16, '56', 4, 56)
    (18, ')', 7, ))
    (20, '/', 25, /)
    (22, '(', 6, ()
    (23, '8', 4, 8)
    (25, '+', 24, +)
    (27, '4', 4, 4)
    (28, ')', 7, ))
    (29, '', 1, )
    (29, '', 1, )
End of token list

<EOT>

check operation priorities being changed by parentheses

Expression (save-tokens):      25 * (8 + 36) / (12 - 7) % 3

Return type:	1
Return value:	1
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '25', 4, 25)
    (3, '*', 25, *)
    (5, '(', 6, ()
    (6, '8', 4, 8)
    (8, '+', 24, +)
    (10, '36', 4, 36)
    (12, ')', 7, ))
    (14, '/', 25, /)
    (16, '(', 6, ()
    (17, '12', 4, 12)
    (20, '-', 24, -)
    (22, '7', 4, 7)
    (23, ')', 7, ))
    (25, '%', 25, %)
    (27, '3', 4, 3)
    (28, '', 1, )
    (28, '', 1, )
End of token list

<EOT>

check non-boolean before boolean operator

Expression:      "xxx" || true

Return type:	<NONE>
Return value:	<NONE>
Error position:	7
Error message:	boolean value expected before operator
Dev message:	<->
Token information:
    Position:	6
    String:	||
    Type:	21
    Value:	||
Value information:
    Type:	2  (string)
    Value:	xxx
Token list:	<NONE>

<EOT>

check non-boolean after boolean operator

Expression:      true && "xxx"

Return type:	<NONE>
Return value:	<NONE>
Error position:	6
Error message:	boolean value expected after operator
Dev message:	<->
Token information:
    Position:	5
    String:	&&
    Type:	22
    Value:	&&
Value information:
    Type:	2  (string)
    Value:	xxx
Token list:	<NONE>

<EOT>

check &&

Expression (save-tokens):     true && true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '&&', 22, &&)
    (8, 'true', 5, <ARRAY>)
    (12, '', 1, )
    (12, '', 1, )
End of token list

<EOT>

check &&

Expression (save-tokens):     false && true

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '&&', 22, &&)
    (9, 'true', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check &&

Expression (save-tokens):     true && false

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '&&', 22, &&)
    (8, 'false', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check &&

Expression (save-tokens):     false && false

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '&&', 22, &&)
    (9, 'false', 5, <ARRAY>)
    (14, '', 1, )
    (14, '', 1, )
End of token list

<EOT>

check ||

Expression (save-tokens):     true || true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '||', 21, ||)
    (8, 'true', 5, <ARRAY>)
    (12, '', 1, )
    (12, '', 1, )
End of token list

<EOT>

check ||

Expression (save-tokens):     false || true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '||', 21, ||)
    (9, 'true', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check ||

Expression (save-tokens):     true || false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '||', 21, ||)
    (8, 'false', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check ||

Expression (save-tokens):     false || false

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '||', 21, ||)
    (9, 'false', 5, <ARRAY>)
    (14, '', 1, )
    (14, '', 1, )
End of token list

<EOT>

check boolean precedence order

Expression (save-tokens):     true || true && false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '||', 21, ||)
    (8, 'true', 5, <ARRAY>)
    (13, '&&', 22, &&)
    (16, 'false', 5, <ARRAY>)
    (21, '', 1, )
    (21, '', 1, )
End of token list

<EOT>

check boolean order with parentheses

Expression (save-tokens):     false && (true || true)

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '&&', 22, &&)
    (9, '(', 6, ()
    (10, 'true', 5, <ARRAY>)
    (15, '||', 21, ||)
    (18, 'true', 5, <ARRAY>)
    (22, ')', 7, ))
    (23, '', 1, )
    (23, '', 1, )
End of token list

<EOT>

check boolean keywords for option not set

Expression:      (true and not false) or false

Return type:	<NONE>
Return value:	<NONE>
Error position:	8
Error message:	unexpected element found
Dev message:	<->
Token information:
    Position:	6
    String:	and
    Type:	2
    Value:	and
Token list:	<NONE>

<EOT>


check boolean keywords for option not set

Expression (boolean-keywords,save-tokens):      (true and not false) or false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '(', 6, ()
    (1, 'true', 5, <ARRAY>)
    (6, 'and', 22, &&)
    (10, 'not', 13, !)
    (14, 'false', 5, <ARRAY>)
    (19, ')', 7, ))
    (21, 'or', 21, ||)
    (24, 'false', 5, <ARRAY>)
    (29, '', 1, )
    (29, '', 1, )
End of token list

<EOT>


check non-boolean after boolean operator

Expression:      true != "xxx"

Return type:	<NONE>
Return value:	<NONE>
Error position:	6
Error message:	boolean value expected after operator
Dev message:	<->
Token information:
    Position:	5
    String:	!=
    Type:	23
    Value:	!=
Value information:
    Type:	2  (string)
    Value:	xxx
Token list:	<NONE>

<EOT>

check !=

Expression (save-tokens):     true != true

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '!=', 23, !=)
    (8, 'true', 5, <ARRAY>)
    (12, '', 1, )
    (12, '', 1, )
End of token list

<EOT>

check !=

Expression (save-tokens):     false != true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '!=', 23, !=)
    (9, 'true', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check !=

Expression (save-tokens):     true != false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '!=', 23, !=)
    (8, 'false', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check !=

Expression (save-tokens):     false != false

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '!=', 23, !=)
    (9, 'false', 5, <ARRAY>)
    (14, '', 1, )
    (14, '', 1, )
End of token list

<EOT>

check ==

Expression (save-tokens):     true == true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '==', 23, ==)
    (8, 'true', 5, <ARRAY>)
    (12, '', 1, )
    (12, '', 1, )
End of token list

<EOT>

check ==

Expression (save-tokens):     false == true

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '==', 23, ==)
    (9, 'true', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check ==

Expression (save-tokens):     true == false

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '==', 23, ==)
    (8, 'false', 5, <ARRAY>)
    (13, '', 1, )
    (13, '', 1, )
End of token list

<EOT>

check ==

Expression (save-tokens):     false == false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '==', 23, ==)
    (9, 'false', 5, <ARRAY>)
    (14, '', 1, )
    (14, '', 1, )
End of token list


<EOT>

check priorities between && and ==

Expression (save-tokens):     false && true == false

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'false', 5, <ARRAY>)
    (6, '&&', 22, &&)
    (9, 'true', 5, <ARRAY>)
    (14, '==', 23, ==)
    (17, 'false', 5, <ARRAY>)
    (22, '', 1, )
    (22, '', 1, )
End of token list

<EOT>

check priorities between || and !=


Expression (save-tokens):     true || false != true

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, 'true', 5, <ARRAY>)
    (5, '||', 21, ||)
    (8, 'false', 5, <ARRAY>)
    (14, '!=', 23, !=)
    (17, 'true', 5, <ARRAY>)
    (21, '', 1, )
    (21, '', 1, )
End of token list

<EOT>

check priorities between && and == with parentheses

Expression (save-tokens):     (false && true) == false

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '(', 6, ()
    (1, 'false', 5, <ARRAY>)
    (7, '&&', 22, &&)
    (10, 'true', 5, <ARRAY>)
    (14, ')', 7, ))
    (16, '==', 23, ==)
    (19, 'false', 5, <ARRAY>)
    (24, '', 1, )
    (24, '', 1, )
End of token list

<EOT>

check priorities between || and != with parentheses


Expression (save-tokens):     (true || false) != true

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '(', 6, ()
    (1, 'true', 5, <ARRAY>)
    (6, '||', 21, ||)
    (9, 'false', 5, <ARRAY>)
    (14, ')', 7, ))
    (16, '!=', 23, !=)
    (19, 'true', 5, <ARRAY>)
    (23, '', 1, )
    (23, '', 1, )
End of token list

<EOT>

check boolean before < comparison


Expression:     true < 5

Return type:	<NONE>
Return value:	<NONE>
Error position:	5
Error message:	string or number expected before operator
Dev message:	<->
Token information:
    Position:	5
    String:	<
    Type:	23
    Value:	<
Value information:
    Type:	3  (bool)
    Value:	t
Token list:	<NONE>

<EOT>

check boolean after < comparison

Expression:     5 < false

Return type:	<NONE>
Return value:	<NONE>
Error position:	2
Error message:	string or number expected after operator
Dev message:	<->
Token information:
    Position:	2
    String:	<
    Type:	23
    Value:	<
Value information:
    Type:	3  (bool)
    Value:	
Token list:	<NONE>

<EOT>

check comparing number to string

Expression:     5 < "xxx"

Return type:	<NONE>
Return value:	<NONE>
Error position:	2
Error message:	values before and after operator need to be the same type
Dev message:	<->
Value information:
    Type:	1  (numeric)
    Value:	5
Value information:
    Type:	2  (string)
    Value:	xxx
Token list:	<NONE>

<EOT>

check string <

Expression (save-tokens):	"a" < "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '<', 23, <)
    (6, '"b"', 3, b)
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check string <

Expression (save-tokens):	"b" < "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"b"', 3, b)
    (4, '<', 23, <)
    (6, '"b"', 3, b)
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check string <

Expression (save-tokens):	"c" < "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"c"', 3, c)
    (4, '<', 23, <)
    (6, '"b"', 3, b)
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check string <=

Expression (save-tokens):	"a" <= "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '<=', 23, <=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string <=

Expression (save-tokens):	"b" <= "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"b"', 3, b)
    (4, '<=', 23, <=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string <=

Expression (save-tokens):	"c" <= "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"c"', 3, c)
    (4, '<=', 23, <=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string ==

Expression (save-tokens):	"a" == "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '==', 23, ==)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string ==

Expression (save-tokens):	"b" == "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"b"', 3, b)
    (4, '==', 23, ==)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string ==

Expression (save-tokens):	"c" == "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"c"', 3, c)
    (4, '==', 23, ==)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string !=

Expression (save-tokens):	"a" != "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '!=', 23, !=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string !=

Expression (save-tokens):	"b" != "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"b"', 3, b)
    (4, '!=', 23, !=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string !=

Expression (save-tokens):	"c" != "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"c"', 3, c)
    (4, '!=', 23, !=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string >=

Expression (save-tokens):	"a" >= "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '>=', 23, >=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string >=

Expression (save-tokens):	"b" >= "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"b"', 3, b)
    (4, '>=', 23, >=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string >=

Expression (save-tokens):	"c" >= "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"c"', 3, c)
    (4, '>=', 23, >=)
    (7, '"b"', 3, b)
    (10, '', 1, )
    (10, '', 1, )
End of token list

<EOT>

check string >

Expression (save-tokens):	"a" > "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '>', 23, >)
    (6, '"b"', 3, b)
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check string >

Expression (save-tokens):	"b" > "b"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"b"', 3, b)
    (4, '>', 23, >)
    (6, '"b"', 3, b)
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check string >

Expression (save-tokens):	"c" > "b"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"c"', 3, c)
    (4, '>', 23, >)
    (6, '"b"', 3, b)
    (9, '', 1, )
    (9, '', 1, )
End of token list

<EOT>

check number <

Expression (save-tokens):	1 < 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1', 4, 1)
    (2, '<', 23, <)
    (4, '2', 4, 2)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check number <

Expression (save-tokens):	2 < 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '2', 4, 2)
    (2, '<', 23, <)
    (4, '2', 4, 2)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check number <

Expression (save-tokens):	3 < 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '3', 4, 3)
    (2, '<', 23, <)
    (4, '2', 4, 2)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check number <=

Expression (save-tokens):	1 <= 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1', 4, 1)
    (2, '<=', 23, <=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number <=

Expression (save-tokens):	2 <= 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '2', 4, 2)
    (2, '<=', 23, <=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number <=

Expression (save-tokens):	3 <= 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '3', 4, 3)
    (2, '<=', 23, <=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number ==

Expression (save-tokens):	1 == 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1', 4, 1)
    (2, '==', 23, ==)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number ==

Expression (save-tokens):	2 == 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '2', 4, 2)
    (2, '==', 23, ==)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number ==

Expression (save-tokens):	3 == 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '3', 4, 3)
    (2, '==', 23, ==)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number !=

Expression (save-tokens):	1 != 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1', 4, 1)
    (2, '!=', 23, !=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number !=

Expression (save-tokens):	2 != 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '2', 4, 2)
    (2, '!=', 23, !=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number !=

Expression (save-tokens):	3 != 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '3', 4, 3)
    (2, '!=', 23, !=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number >=

Expression (save-tokens):	1 >= 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1', 4, 1)
    (2, '>=', 23, >=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number >=

Expression (save-tokens):	2 >= 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '2', 4, 2)
    (2, '>=', 23, >=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number >=

Expression (save-tokens):	3 >= 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '3', 4, 3)
    (2, '>=', 23, >=)
    (5, '2', 4, 2)
    (6, '', 1, )
    (6, '', 1, )
End of token list

<EOT>

check number >

Expression (save-tokens):	1 > 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '1', 4, 1)
    (2, '>', 23, >)
    (4, '2', 4, 2)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check number >

Expression (save-tokens):	2 > 2

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '2', 4, 2)
    (2, '>', 23, >)
    (4, '2', 4, 2)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check number >

Expression (save-tokens):	3 > 2

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '3', 4, 3)
    (2, '>', 23, >)
    (4, '2', 4, 2)
    (5, '', 1, )
    (5, '', 1, )
End of token list

<EOT>

check if =~ has string before operand

Expression:	false =~ "XXX"

Return type:	<NONE>
Return value:	<NONE>
Error position:	7
Error message:	string expected before operator
Dev message:	<->
Token information:
    Position:	6
    String:	=~
    Type:	23
    Value:	=~
Value information:
    Type:	3  (bool)
    Value:	
Token list:	<NONE>

<EOT>

check if =~ has string after operand

Expression:	"XXX" !~ true

Return type:	<NONE>
Return value:	<NONE>
Error position:	7
Error message:	string expected after operator
Dev message:	<->
Token information:
    Position:	6
    String:	!~
    Type:	23
    Value:	!~
Value information:
    Type:	3  (bool)
    Value:	t
Token list:	<NONE>

<EOT>

check if =~ pattern is valid
(note:  format of error message may vary depending on perl version, system, etc)

Expression:	"XXX" =~ ".[hjk"

Return type:	<NONE>
Return value:	<NONE>
Error position:	7
Error message:	/.[hjk/: unmatched [] in regexp
Dev message:	<->
Pattern:  /.[hjk/
/.[hjk/: unmatched [] in regexp at expr_eval.pm line 1735.
Token list:	<NONE>

<EOT>

check if =~ matched

Expression (save-tokens):	"678XXX" =~ "67[789]X"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"678XXX"', 3, 678XXX)
    (9, '=~', 23, =~)
    (12, '"67[789]X"', 3, 67[789]X)
    (22, '', 1, )
    (22, '', 1, )
End of token list

<EOT>

check if =~ matched

Expression (save-tokens):	"675XXX" =~ "67[789]X"

Return type:	3
Return value:	
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"675XXX"', 3, 675XXX)
    (9, '=~', 23, =~)
    (12, '"67[789]X"', 3, 67[789]X)
    (22, '', 1, )
    (22, '', 1, )
End of token list

<EOT>

check if !~ matched

Expression (save-tokens):	"675XXX" !~ "67[789]X"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"675XXX"', 3, 675XXX)
    (9, '!~', 23, !~)
    (12, '"67[789]X"', 3, 67[789]X)
    (22, '', 1, )
    (22, '', 1, )
End of token list

<EOT>

check precedence of + and =~

Expression (save-tokens):	"67" + "8XXX" =~ "67[7" + "89]X"

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"67"', 3, 67)
    (5, '+', 24, +)
    (7, '"8XXX"', 3, 8XXX)
    (14, '=~', 23, =~)
    (17, '"67[7"', 3, 67[7)
    (24, '+', 24, +)
    (26, '"89]X"', 3, 89]X)
    (32, '', 1, )
    (32, '', 1, )
End of token list

<EOT>

check compound expression for checking priorities

Expression (save-tokens):     "a" + 2 * 4 < "b" + 2 * 3

Return type:	3
Return value:	t
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"a"', 3, a)
    (4, '+', 24, +)
    (6, '2', 4, 2)
    (8, '*', 25, *)
    (10, '4', 4, 4)
    (12, '<', 23, <)
    (14, '"b"', 3, b)
    (18, '+', 24, +)
    (20, '2', 4, 2)
    (22, '*', 25, *)
    (24, '3', 4, 3)
    (25, '', 1, )
    (25, '', 1, )
End of token list

<EOT>

check for special binary operator without operator function

Expression (token-function):      "XXX" %binary-operator% "YYY"

Return type:	<NONE>
Return value:	<NONE>
Error position:	22
Error message:	operator not evaluated by function
Dev message:	<->
No function specified.
Token information:
    Position:	6
    String:	%binary-operator%
    Type:	70
    Value:	%binary-operator%
Token list:	<NONE>

<EOT>

check for special binary operator

Expression (token-function,operator-function,save-tokens):      "XXX" %binary-operator% "YYY"

Return type:	2
Return value:	(YYY,XXX)
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"XXX"', 3, XXX)
    (6, '%binary-operator%', 70, %binary-operator%)
    (24, '"YYY"', 3, YYY)
    (29, '', 1, )
    (29, '', 1, )
End of token list

<EOT>

check for special binary operator acting on non-string

Expression (token-function,operator-function):      1000 %binary-operator% "YYY"

Return type:	<NONE>
Return value:	<NONE>
Error position:	21
Error message:	string expected before operator
Dev message:	<->
function is code
Token information:
    Position:	5
    String:	%binary-operator%
    Type:	70
    Value:	%binary-operator%
Token list:	<NONE>

<EOT>

check for operator function error
(note:  format of error message may vary depending on perl version, system, etc)

Expression (token-function,operator-function):      "XXX" %binary-operator-error% "YYY"

Return type:	<NONE>
Return value:	<NONE>
Error position:	28
Error message:	system error
Dev message:	<->
function is code
Token information:
    Position:	6
    String:	%binary-operator-error%
    Type:	70
    Value:	%binary-operator-error%
Illegal division by zero at test.pl line 357.
Token list:	<NONE>

<EOT>

check for special binary operator precedence

Expression (token-function,operator-function,save-tokens):      "AAA" + "XXX" %binary-operator% "YYY" + "ZZZ"

Return type:	2
Return value:	AAA(YYY,XXX)ZZZ
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '"AAA"', 3, AAA)
    (6, '+', 24, +)
    (8, '"XXX"', 3, XXX)
    (14, '%binary-operator%', 70, %binary-operator%)
    (32, '"YYY"', 3, YYY)
    (38, '+', 24, +)
    (40, '"ZZZ"', 3, ZZZ)
    (45, '', 1, )
    (45, '', 1, )
End of token list

<EOT>

check for special binary operator precedence with parentheses

Expression (token-function,operator-function,save-tokens):      ("AAA" + "XXX") %binary-operator% ("YYY" + "ZZZ")

Return type:	2
Return value:	(YYYZZZ,AAAXXX)
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '(', 6, ()
    (1, '"AAA"', 3, AAA)
    (7, '+', 24, +)
    (9, '"XXX"', 3, XXX)
    (14, ')', 7, ))
    (16, '%binary-operator%', 70, %binary-operator%)
    (34, '(', 6, ()
    (35, '"YYY"', 3, YYY)
    (41, '+', 24, +)
    (43, '"ZZZ"', 3, ZZZ)
    (48, ')', 7, ))
    (49, '', 1, )
    (49, '', 1, )
End of token list

<EOT>

check syntax checking only option

Expression (syntax-check-only,save-tokens):      25 * 6 + 36 / 12 - 8 % 3

Return type:	<NONE>
Return value:	<NONE>
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '25', 4, 25)
    (3, '*', 25, *)
    (5, '6', 4, 6)
    (7, '+', 24, +)
    (9, '36', 4, 36)
    (12, '/', 25, /)
    (14, '12', 4, 12)
    (17, '-', 24, -)
    (19, '8', 4, 8)
    (21, '%', 25, %)
    (23, '3', 4, 3)
    (24, '', 1, )
    (24, '', 1, )
End of token list

<EOT>

check token list replay

Expression (replay-tokens):      25 * 6 + 36 / 12 - 8 % 3

Return type:	1
Return value:	151
Error position:	<NONE>
Error message:	<NONE>
Dev message:	<NONE>
Token list:
    (0, '25', 4, 25)
    (3, '*', 25, *)
    (5, '6', 4, 6)
    (7, '+', 24, +)
    (9, '36', 4, 36)
    (12, '/', 25, /)
    (14, '12', 4, 12)
    (17, '-', 24, -)
    (19, '8', 4, 8)
    (21, '%', 25, %)
    (23, '3', 4, 3)
    (24, '', 1, )
    (24, '', 1, )
End of token list

<EOT>
EOF
}


############################# Copyright #######################################

# Copyright (c) 2001 Scott Luebking. All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Perl itself.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

###############################################################################

 
