# -*- mode: perl; -*-

use strict;
use warnings;

use Scalar::Util qw< refaddr >;
use Test::More tests => 51951;

my @classes;

BEGIN {
    @classes = qw< Math::BigInt Math::BigRat Math::BigFloat >;
    for my $class (@classes) {
        use_ok($class);
    }
}

while (<DATA>) {
    s/(^|\s+)#.*$//;            # remove comments
    s/^\s+//;                   # remove leading whitespace
    s/\s+$//;                   # remove trailing whitespace
    next unless length;         # skip empty lines

    my ($nval, $kval, $outval) = split /:/;

    for my $class (@classes) {
        for my $scalar_context (0, 1) {
            for my $instance_method (0, 1) {
                for my $n_is_scalar (0, 1) {
                    for my $k_is_scalar (0, 1) {

                        my ($n, $k, $got, @got);

                        # We cannot use an instance method when the first
                        # operand is scalar.

                        next if $instance_method && $n_is_scalar;

                        # Build test.

                        my $test;

                        $test .= $n_is_scalar
                          ? qq|\$n = "$nval";|
                          : qq|\$n = $class -> new("$nval");|;

                        $test .= $k_is_scalar
                          ? qq| \$k = "$kval";|
                          : qq| \$k = $class -> new("$kval");|;

                        $test .= $scalar_context
                          ? qq| \$got =|
                          : qq| \@got =|;

                        $test .= $instance_method
                          ? qq| \$n -> bperm(\$k);|
                          : qq| $class -> bperm(\$n, \$k);|;

                        # Display tests.

                        note("\n$test\n\n");

                        subtest $test, sub {

                            eval $test;
                            is($@, "", '$@ is empty');

                            unless ($scalar_context) {

                                # Check number of output arguments.

                                cmp_ok(scalar(@got), '==', 1,
                                       "got one output arg");

                                $got = $got[0];
                            }

                            # Check output.

                            is(ref($got), $class,
                               "output arg is a $class");

                            is($got, $outval,
                               "output arg has the right value");

                            unless ($n_is_scalar) {
                                is(refaddr($got), refaddr($n),
                                   "the output is the first operand");
                            }

                            # Check the second operand.

                            if ($k_is_scalar) {

                                is(ref($k), '',
                                   "second input arg is still a scalar");

                                is($k, $kval,
                                   "second input arg is unmodified");

                            } else {

                                is(ref($k), $class,
                                   "second input arg is still a $class");

                                is($k, $kval,
                                   "second input arg is unmodified");
                            }
                        };
                    }
                }
            }
        }
    }
}

__DATA__

NaN:NaN:NaN
NaN:3:NaN
3:NaN:NaN

inf:-inf:NaN
inf:-1:NaN
inf:0:1
inf:1:inf
inf:2:inf
inf:inf:NaN

0:inf:NaN
1:inf:NaN
2:inf:NaN

0:-1:NaN
0:0:1
0:1:NaN

1:-1:NaN
1:0:1
1:1:1
1:2:NaN

2:-1:NaN
2:0:1
2:1:2
2:2:2
2:3:NaN

3:-1:NaN
3:0:1
3:1:3
3:2:6
3:3:6
3:4:NaN

4:-1:NaN
4:0:1
4:1:4
4:2:12
4:3:24
4:4:24
4:5:NaN

5:-1:NaN
5:0:1
5:1:5
5:2:20
5:3:60
5:4:120
5:5:120
5:6:NaN

6:-1:NaN
6:0:1
6:1:6
6:2:30
6:3:120
6:4:360
6:5:720
6:6:720
6:7:NaN

7:-1:NaN
7:0:1
7:1:7
7:2:42
7:3:210
7:4:840
7:5:2520
7:6:5040
7:7:5040
7:8:NaN

8:-1:NaN
8:0:1
8:1:8
8:2:56
8:3:336
8:4:1680
8:5:6720
8:6:20160
8:7:40320
8:8:40320
8:9:NaN

9:-1:NaN
9:0:1
9:1:9
9:2:72
9:3:504
9:4:3024
9:5:15120
9:6:60480
9:7:181440
9:8:362880
9:9:362880
9:10:NaN

10:-1:NaN
10:0:1
10:1:10
10:2:90
10:3:720
10:4:5040
10:5:30240
10:6:151200
10:7:604800
10:8:1814400
10:9:3628800
10:10:3628800
10:11:NaN

11:-1:NaN
11:0:1
11:1:11
11:2:110
11:3:990
11:4:7920
11:5:55440
11:6:332640
11:7:1663200
11:8:6652800
11:9:19958400
11:10:39916800
11:11:39916800
11:12:NaN

12:-1:NaN
12:0:1
12:1:12
12:2:132
12:3:1320
12:4:11880
12:5:95040
12:6:665280
12:7:3991680
12:8:19958400
12:9:79833600
12:10:239500800
12:11:479001600
12:12:479001600
12:13:NaN

13:-1:NaN
13:0:1
13:1:13
13:2:156
13:3:1716
13:4:17160
13:5:154440
13:6:1235520
13:7:8648640
13:8:51891840
13:9:259459200
13:10:1037836800
13:11:3113510400
13:12:6227020800
13:13:6227020800
13:14:NaN

14:-1:NaN
14:0:1
14:1:14
14:2:182
14:3:2184
14:4:24024
14:5:240240
14:6:2162160
14:7:17297280
14:8:121080960
14:9:726485760
14:10:3632428800
14:11:14529715200
14:12:43589145600
14:13:87178291200
14:14:87178291200
14:15:NaN

15:-1:NaN
15:0:1
15:1:15
15:2:210
15:3:2730
15:4:32760
15:5:360360
15:6:3603600
15:7:32432400
15:8:259459200
15:9:1816214400
15:10:10897286400
15:11:54486432000
15:12:217945728000
15:13:653837184000
15:14:1307674368000
15:15:1307674368000
15:16:NaN

16:-1:NaN
16:0:1
16:1:16
16:2:240
16:3:3360
16:4:43680
16:5:524160
16:6:5765760
16:7:57657600
16:8:518918400
16:9:4151347200
16:10:29059430400
16:11:174356582400
16:12:871782912000
16:13:3487131648000
16:14:10461394944000
16:15:20922789888000
16:16:20922789888000
16:17:NaN

17:-1:NaN
17:0:1
17:1:17
17:2:272
17:3:4080
17:4:57120
17:5:742560
17:6:8910720
17:7:98017920
17:8:980179200
17:9:8821612800
17:10:70572902400
17:11:494010316800
17:12:2964061900800
17:13:14820309504000
17:14:59281238016000
17:15:177843714048000
17:16:355687428096000
17:17:355687428096000
17:18:NaN

18:-1:NaN
18:0:1
18:1:18
18:2:306
18:3:4896
18:4:73440
18:5:1028160
18:6:13366080
18:7:160392960
18:8:1764322560
18:9:17643225600
18:10:158789030400
18:11:1270312243200
18:12:8892185702400
18:13:53353114214400
18:14:266765571072000
18:15:1067062284288000
18:16:3201186852864000
18:17:6402373705728000
18:18:6402373705728000
18:19:NaN

19:-1:NaN
19:0:1
19:1:19
19:2:342
19:3:5814
19:4:93024
19:5:1395360
19:6:19535040
19:7:253955520
19:8:3047466240
19:9:33522128640
19:10:335221286400
19:11:3016991577600
19:12:24135932620800
19:13:168951528345600
19:14:1013709170073600
19:15:5068545850368000
19:16:20274183401472000
19:17:60822550204416000
19:18:121645100408832000
19:19:121645100408832000
19:20:NaN

20:-1:NaN
20:0:1
20:1:20
20:2:380
20:3:6840
20:4:116280
20:5:1860480
20:6:27907200
20:7:390700800
20:8:5079110400
20:9:60949324800
20:10:670442572800
20:11:6704425728000
20:12:60339831552000
20:13:482718652416000
20:14:3379030566912000
20:15:20274183401472000
20:16:101370917007360000
20:17:405483668029440000
20:18:1216451004088320000
20:19:2432902008176640000
20:20:2432902008176640000
20:21:NaN

21:-1:NaN
21:0:1
21:1:21
21:2:420
21:3:7980
21:4:143640
21:5:2441880
21:6:39070080
21:7:586051200
21:8:8204716800
21:9:106661318400
21:10:1279935820800
21:11:14079294028800
21:12:140792940288000
21:13:1267136462592000
21:14:10137091700736000
21:15:70959641905152000
21:16:425757851430912000
21:17:2128789257154560000
21:18:8515157028618240000
21:19:25545471085854720000
21:20:51090942171709440000
21:21:51090942171709440000
21:22:NaN

22:-1:NaN
22:0:1
22:1:22
22:2:462
22:3:9240
22:4:175560
22:5:3160080
22:6:53721360
22:7:859541760
22:8:12893126400
22:9:180503769600
22:10:2346549004800
22:11:28158588057600
22:12:309744468633600
22:13:3097444686336000
22:14:27877002177024000
22:15:223016017416192000
22:16:1561112121913344000
22:17:9366672731480064000
22:18:46833363657400320000
22:19:187333454629601280000
22:20:562000363888803840000
22:21:1124000727777607680000
22:22:1124000727777607680000
22:23:NaN

23:-1:NaN
23:0:1
23:1:23
23:2:506
23:3:10626
23:4:212520
23:5:4037880
23:6:72681840
23:7:1235591280
23:8:19769460480
23:9:296541907200
23:10:4151586700800
23:11:53970627110400
23:12:647647525324800
23:13:7124122778572800
23:14:71241227785728000
23:15:641171050071552000
23:16:5129368400572416000
23:17:35905578804006912000
23:18:215433472824041472000
23:19:1077167364120207360000
23:20:4308669456480829440000
23:21:12926008369442488320000
23:22:25852016738884976640000
23:23:25852016738884976640000
23:24:NaN

24:-1:NaN
24:0:1
24:1:24
24:2:552
24:3:12144
24:4:255024
24:5:5100480
24:6:96909120
24:7:1744364160
24:8:29654190720
24:9:474467051520
24:10:7117005772800
24:11:99638080819200
24:12:1295295050649600
24:13:15543540607795200
24:14:170978946685747200
24:15:1709789466857472000
24:16:15388105201717248000
24:17:123104841613737984000
24:18:861733891296165888000
24:19:5170403347776995328000
24:20:25852016738884976640000
24:21:103408066955539906560000
24:22:310224200866619719680000
24:23:620448401733239439360000
24:24:620448401733239439360000
24:25:NaN

25:-1:NaN
25:0:1
25:1:25
25:2:600
25:3:13800
25:4:303600
25:5:6375600
25:6:127512000
25:7:2422728000
25:8:43609104000
25:9:741354768000
25:10:11861676288000
25:11:177925144320000
25:12:2490952020480000
25:13:32382376266240000
25:14:388588515194880000
25:15:4274473667143680000
25:16:42744736671436800000
25:17:384702630042931200000
25:18:3077621040343449600000
25:19:21543347282404147200000
25:20:129260083694424883200000
25:21:646300418472124416000000
25:22:2585201673888497664000000
25:23:7755605021665492992000000
25:24:15511210043330985984000000
25:25:15511210043330985984000000
25:26:NaN

26:-1:NaN
26:0:1
26:1:26
26:2:650
26:3:15600
26:4:358800
26:5:7893600
26:6:165765600
26:7:3315312000
26:8:62990928000
26:9:1133836704000
26:10:19275223968000
26:11:308403583488000
26:12:4626053752320000
26:13:64764752532480000
26:14:841941782922240000
26:15:10103301395066880000
26:16:111136315345735680000
26:17:1111363153457356800000
26:18:10002268381116211200000
26:19:80018147048929689600000
26:20:560127029342507827200000
26:21:3360762176055046963200000
26:22:16803810880275234816000000
26:23:67215243521100939264000000
26:24:201645730563302817792000000
26:25:403291461126605635584000000
26:26:403291461126605635584000000
26:27:NaN

27:-1:NaN
27:0:1
27:1:27
27:2:702
27:3:17550
27:4:421200
27:5:9687600
27:6:213127200
27:7:4475671200
27:8:89513424000
27:9:1700755056000
27:10:30613591008000
27:11:520431047136000
27:12:8326896754176000
27:13:124903451312640000
27:14:1748648318376960000
27:15:22732428138900480000
27:16:272789137666805760000
27:17:3000680514334863360000
27:18:30006805143348633600000
27:19:270061246290137702400000
27:20:2160489970321101619200000
27:21:15123429792247711334400000
27:22:90740578753486268006400000
27:23:453702893767431340032000000
27:24:1814811575069725360128000000
27:25:5444434725209176080384000000
27:26:10888869450418352160768000000
27:27:10888869450418352160768000000
27:28:NaN

28:-1:NaN
28:0:1
28:1:28
28:2:756
28:3:19656
28:4:491400
28:5:11793600
28:6:271252800
28:7:5967561600
28:8:125318793600
28:9:2506375872000
28:10:47621141568000
28:11:857180548224000
28:12:14572069319808000
28:13:233153109116928000
28:14:3497296636753920000
28:15:48962152914554880000
28:16:636507987889213440000
28:17:7638095854670561280000
28:18:84019054401376174080000
28:19:840190544013761740800000
28:20:7561714896123855667200000
28:21:60493719168990845337600000
28:22:423456034182935917363200000
28:23:2540736205097615504179200000
28:24:12703681025488077520896000000
28:25:50814724101952310083584000000
28:26:152444172305856930250752000000
28:27:304888344611713860501504000000
28:28:304888344611713860501504000000
28:29:NaN

29:-1:NaN
29:0:1
29:1:29
29:2:812
29:3:21924
29:4:570024
29:5:14250600
29:6:342014400
29:7:7866331200
29:8:173059286400
29:9:3634245014400
29:10:72684900288000
29:11:1381013105472000
29:12:24858235898496000
29:13:422590010274432000
29:14:6761440164390912000
29:15:101421602465863680000
29:16:1419902434522091520000
29:17:18458731648787189760000
29:18:221504779785446277120000
29:19:2436552577639909048320000
29:20:24365525776399090483200000
29:21:219289731987591814348800000
29:22:1754317855900734514790400000
29:23:12280224991305141603532800000
29:24:73681349947830849621196800000
29:25:368406749739154248105984000000
29:26:1473626998956616992423936000000
29:27:4420880996869850977271808000000
29:28:8841761993739701954543616000000
29:29:8841761993739701954543616000000
29:30:NaN

30:-1:NaN
30:0:1
30:1:30
30:2:870
30:3:24360
30:4:657720
30:5:17100720
30:6:427518000
30:7:10260432000
30:8:235989936000
30:9:5191778592000
30:10:109027350432000
30:11:2180547008640000
30:12:41430393164160000
30:13:745747076954880000
30:14:12677700308232960000
30:15:202843204931727360000
30:16:3042648073975910400000
30:17:42597073035662745600000
30:18:553761949463615692800000
30:19:6645143393563388313600000
30:20:73096577329197271449600000
30:21:730965773291972714496000000
30:22:6578691959627754430464000000
30:23:52629535677022035443712000000
30:24:368406749739154248105984000000
30:25:2210440498434925488635904000000
30:26:11052202492174627443179520000000
30:27:44208809968698509772718080000000
30:28:132626429906095529318154240000000
30:29:265252859812191058636308480000000
30:30:265252859812191058636308480000000
30:31:NaN

31:-1:NaN
31:0:1
31:1:31
31:2:930
31:3:26970
31:4:755160
31:5:20389320
31:6:530122320
31:7:13253058000
31:8:318073392000
31:9:7315688016000
31:10:160945136352000
31:11:3379847863392000
31:12:67596957267840000
31:13:1284342188088960000
31:14:23118159385601280000
31:15:393008709555221760000
31:16:6288139352883548160000
31:17:94322090293253222400000
31:18:1320509264105545113600000
31:19:17166620433372086476800000
31:20:205999445200465037721600000
31:21:2265993897205115414937600000
31:22:22659938972051154149376000000
31:23:203939450748460387344384000000
31:24:1631515605987683098755072000000
31:25:11420609241913781691285504000000
31:26:68523655451482690147713024000000
31:27:342618277257413450738565120000000
31:28:1370473109029653802954260480000000
31:29:4111419327088961408862781440000000
31:30:8222838654177922817725562880000000
31:31:8222838654177922817725562880000000
31:32:NaN

32:-1:NaN
32:0:1
32:1:32
32:2:992
32:3:29760
32:4:863040
32:5:24165120
32:6:652458240
32:7:16963914240
32:8:424097856000
32:9:10178348544000
32:10:234102016512000
32:11:5150244363264000
32:12:108155131628544000
32:13:2163102632570880000
32:14:41098950018846720000
32:15:739781100339240960000
32:16:12576278705767096320000
32:17:201220459292273541120000
32:18:3018306889384103116800000
32:19:42256296451377443635200000
32:20:549331853867906767257600000
32:21:6591982246414881207091200000
32:22:72511804710563693278003200000
32:23:725118047105636932780032000000
32:24:6526062423950732395020288000000
32:25:52208499391605859160162304000000
32:26:365459495741241014121136128000000
32:27:2192756974447446084726816768000000
32:28:10963784872237230423634083840000000
32:29:43855139488948921694536335360000000
32:30:131565418466846765083609006080000000
32:31:263130836933693530167218012160000000
32:32:263130836933693530167218012160000000
32:33:NaN

33:-1:NaN
33:0:1
33:1:33
33:2:1056
33:3:32736
33:4:982080
33:5:28480320
33:6:797448960
33:7:21531121920
33:8:559809169920
33:9:13995229248000
33:10:335885501952000
33:11:7725366544896000
33:12:169958063987712000
33:13:3569119343741952000
33:14:71382386874839040000
33:15:1356265350621941760000
33:16:24412776311194951680000
33:17:415017197290314178560000
33:18:6640275156645026856960000
33:19:99604127349675402854400000
33:20:1394457782895455639961600000
33:21:18127951177640923319500800000
33:22:217535414131691079834009600000
33:23:2392889555448601878174105600000
33:24:23928895554486018781741056000000
33:25:215360059990374169035669504000000
33:26:1722880479922993352285356032000000
33:27:12060163359460953465997492224000000
33:28:72360980156765720795984953344000000
33:29:361804900783828603979924766720000000
33:30:1447219603135314415919699066880000000
33:31:4341658809405943247759097200640000000
33:32:8683317618811886495518194401280000000
33:33:8683317618811886495518194401280000000
33:34:NaN

34:-1:NaN
34:0:1
34:1:34
34:2:1122
34:3:35904
34:4:1113024
34:5:33390720
34:6:968330880
34:7:27113264640
34:8:732058145280
34:9:19033511777280
34:10:475837794432000
34:11:11420107066368000
34:12:262662462526464000
34:13:5778574175582208000
34:14:121350057687226368000
34:15:2427001153744527360000
34:16:46113021921146019840000
34:17:830034394580628357120000
34:18:14110584707870682071040000
34:19:225769355325930913136640000
34:20:3386540329888963697049600000
34:21:47411564618445491758694400000
34:22:616350340039791392863027200000
34:23:7396204080477496714356326400000
34:24:81358244885252463857919590400000
34:25:813582448852524638579195904000000
34:26:7322242039672721747212763136000000
34:27:58577936317381773977702105088000000
34:28:410045554221672417843914735616000000
34:29:2460273325330034507063488413696000000
34:30:12301366626650172535317442068480000000
34:31:49205466506600690141269768273920000000
34:32:147616399519802070423809304821760000000
34:33:295232799039604140847618609643520000000
34:34:295232799039604140847618609643520000000
34:35:NaN

35:-1:NaN
35:0:1
35:1:35
35:2:1190
35:3:39270
35:4:1256640
35:5:38955840
35:6:1168675200
35:7:33891580800
35:8:948964262400
35:9:25622035084800
35:10:666172912204800
35:11:16654322805120000
35:12:399703747322880000
35:13:9193186188426240000
35:14:202250096145377280000
35:15:4247252019052922880000
35:16:84945040381058457600000
35:17:1613955767240110694400000
35:18:29051203810321992499200000
35:19:493870464775473872486400000
35:20:7901927436407581959782400000
35:21:118528911546113729396736000000
35:22:1659404761645592211554304000000
35:23:21572261901392698750205952000000
35:24:258867142816712385002471424000000
35:25:2847538570983836235027185664000000
35:26:28475385709838362350271856640000000
35:27:256278471388545261152446709760000000
35:28:2050227771108362089219573678080000000
35:29:14351594397758534624537015746560000000
35:30:86109566386551207747222094479360000000
35:31:430547831932756038736110472396800000000
35:32:1722191327731024154944441889587200000000
35:33:5166573983193072464833325668761600000000
35:34:10333147966386144929666651337523200000000
35:35:10333147966386144929666651337523200000000
35:36:NaN

36:-1:NaN
36:0:1
36:1:36
36:2:1260
36:3:42840
36:4:1413720
36:5:45239040
36:6:1402410240
36:7:42072307200
36:8:1220096908800
36:9:34162713446400
36:10:922393263052800
36:11:23982224839372800
36:12:599555620984320000
36:13:14389334903623680000
36:14:330954702783344640000
36:15:7281003461233582080000
36:16:152901072685905223680000
36:17:3058021453718104473600000
36:18:58102407620643984998400000
36:19:1045843337171591729971200000
36:20:17779336731917059409510400000
36:21:284469387710672950552166400000
36:22:4267040815660094258282496000000
36:23:59738571419241319615954944000000
36:24:776601428450137155007414272000000
36:25:9319217141401645860088971264000000
36:26:102511388555418104460978683904000000
36:27:1025113885554181044609786839040000000
36:28:9226024969987629401488081551360000000
36:29:73808199759901035211904652410880000000
36:30:516657398319307246483332566876160000000
36:31:3099944389915843478899995401256960000000
36:32:15499721949579217394499977006284800000000
36:33:61998887798316869577999908025139200000000
36:34:185996663394950608733999724075417600000000
36:35:371993326789901217467999448150835200000000
36:36:371993326789901217467999448150835200000000
36:37:NaN

37:-1:NaN
37:0:1
37:1:37
37:2:1332
37:3:46620
37:4:1585080
37:5:52307640
37:6:1673844480
37:7:51889178880
37:8:1556675366400
37:9:45143585625600
37:10:1264020397516800
37:11:34128550732953600
37:12:887342319056793600
37:13:22183557976419840000
37:14:532405391434076160000
37:15:12245324002983751680000
37:16:269397128065642536960000
37:17:5657339689378493276160000
37:18:113146793787569865523200000
37:19:2149789081963827444940800000
37:20:38696203475348894008934400000
37:21:657835459080931198151884800000
37:22:10525367345294899170430156800000
37:23:157880510179423487556452352000000
37:24:2210327142511928825790332928000000
37:25:28734252852655074735274328064000000
37:26:344811034231860896823291936768000000
37:27:3792921376550469865056211304448000000
37:28:37929213765504698650562113044480000000
37:29:341362923889542287855059017400320000000
37:30:2730903391116338302840472139202560000000
37:31:19116323737814368119883304974417920000000
37:32:114697942426886208719299829846507520000000
37:33:573489712134431043596499149232537600000000
37:34:2293958848537724174385996596930150400000000
37:35:6881876545613172523157989790790451200000000
37:36:13763753091226345046315979581580902400000000
37:37:13763753091226345046315979581580902400000000
37:38:NaN

38:-1:NaN
38:0:1
38:1:38
38:2:1406
38:3:50616
38:4:1771560
38:5:60233040
38:6:1987690320
38:7:63606090240
38:8:1971788797440
38:9:59153663923200
38:10:1715456253772800
38:11:48032775105638400
38:12:1296884927852236800
38:13:33719008124158156800
38:14:842975203103953920000
38:15:20231404874494894080000
38:16:465322312113382563840000
38:17:10237090866494416404480000
38:18:214978908196382744494080000
38:19:4299578163927654889881600000
38:20:81691985114625442907750400000
38:21:1470455732063257972339507200000
38:22:24997747445075385529771622400000
38:23:399963959121206168476345958400000
38:24:5999459386818092527145189376000000
38:25:83992431415453295380032651264000000
38:26:1091901608400892839940424466432000000
38:27:13102819300810714079285093597184000000
38:28:144131012308917854872136029569024000000
38:29:1441310123089178548721360295690240000000
38:30:12971791107802606938492242661212160000000
38:31:103774328862420855507937941289697280000000
38:32:726420302036945988555565589027880960000000
38:33:4358521812221675931333393534167285760000000
38:34:21792609061108379656666967670836428800000000
38:35:87170436244433518626667870683345715200000000
38:36:261511308733300555880003612050037145600000000
38:37:523022617466601111760007224100074291200000000
38:38:523022617466601111760007224100074291200000000
38:39:NaN

39:-1:NaN
39:0:1
39:1:39
39:2:1482
39:3:54834
39:4:1974024
39:5:69090840
39:6:2349088560
39:7:77519922480
39:8:2480637519360
39:9:76899763100160
39:10:2306992893004800
39:11:66902793897139200
39:12:1873278229119897600
39:13:50578512186237235200
39:14:1315041316842168115200
39:15:32876032921054202880000
39:16:789024790105300869120000
39:17:18147570172421919989760000
39:18:399246543793282239774720000
39:19:8384177419658927035269120000
39:20:167683548393178540705382400000
39:21:3185987419470392273402265600000
39:22:57347773550467060921240780800000
39:23:974912150357940035661093273600000
39:24:15598594405727040570577492377600000
39:25:233978916085905608558662385664000000
39:26:3275704825202678519821273399296000000
39:27:42584162727634820757676554190848000000
39:28:511009952731617849092118650290176000000
39:29:5621109480047796340013305153191936000000
39:30:56211094800477963400133051531919360000000
39:31:505899853204301670601197463787274240000000
39:32:4047198825634413364809579710298193920000000
39:33:28330391779440893553667057972087357440000000
39:34:169982350676645361322002347832524144640000000
39:35:849911753383226806610011739162620723200000000
39:36:3399647013532907226440046956650482892800000000
39:37:10198941040598721679320140869951448678400000000
39:38:20397882081197443358640281739902897356800000000
39:39:20397882081197443358640281739902897356800000000
39:40:NaN

40:-1:NaN
40:0:1
40:1:40
40:2:1560
40:3:59280
40:4:2193360
40:5:78960960
40:6:2763633600
40:7:93963542400
40:8:3100796899200
40:9:99225500774400
40:10:3075990524006400
40:11:92279715720192000
40:12:2676111755885568000
40:13:74931129164795904000
40:14:2023140487449489408000
40:15:52601652673686724608000
40:16:1315041316842168115200000
40:17:31560991604212034764800000
40:18:725902806896876799590400000
40:19:15969861751731289590988800000
40:20:335367096786357081410764800000
40:21:6707341935727141628215296000000
40:22:127439496778815690936090624000000
40:23:2293910942018682436849631232000000
40:24:38996486014317601426443730944000000
40:25:623943776229081622823099695104000000
40:26:9359156643436224342346495426560000000
40:27:131028193008107140792850935971840000000
40:28:1703366509105392830307062167633920000000
40:29:20440398109264713963684746011607040000000
40:30:224844379201911853600532206127677440000000
40:31:2248443792019118536005322061276774400000000
40:32:20235994128172066824047898551490969600000000
40:33:161887953025376534592383188411927756800000000
40:34:1133215671177635742146682318883494297600000000
40:35:6799294027065814452880093913300965785600000000
40:36:33996470135329072264400469566504828928000000000
40:37:135985880541316289057601878266019315712000000000
40:38:407957641623948867172805634798057947136000000000
40:39:815915283247897734345611269596115894272000000000
40:40:815915283247897734345611269596115894272000000000
40:41:NaN

41:-1:NaN
41:0:1
41:1:41
41:2:1640
41:3:63960
41:4:2430480
41:5:89927760
41:6:3237399360
41:7:113308977600
41:8:3852505238400
41:9:127132672867200
41:10:4068245531750400
41:11:126115611484262400
41:12:3783468344527872000
41:13:109720581991308288000
41:14:3072176295756632064000
41:15:82948759985429065728000
41:16:2156667759621155708928000
41:17:53916693990528892723200000
41:18:1294000655772693425356800000
41:19:29762015082771948783206400000
41:20:654764331820982873230540800000
41:21:13750050968240640337841356800000
41:22:275001019364812806756827136000000
41:23:5225019367931443328379715584000000
41:24:94050348622765979910834880512000000
41:25:1598855926587021658484192968704000000
41:26:25581694825392346535747087499264000000
41:27:383725422380885198036206312488960000000
41:28:5372155913332392772506888374845440000000
41:29:69838026873321106042589548872990720000000
41:30:838056322479853272511074586475888640000000
41:31:9218619547278385997621820451234775040000000
41:32:92186195472783859976218204512347750400000000
41:33:829675759255054739785963840611129753600000000
41:34:6637406074040437918287710724889038028800000000
41:35:46461842518283065428013975074223266201600000000
41:36:278771055109698392568083850445339597209600000000
41:37:1393855275548491962840419252226697986048000000000
41:38:5575421102193967851361677008906791944192000000000
41:39:16726263306581903554085031026720375832576000000000
41:40:33452526613163807108170062053440751665152000000000
41:41:33452526613163807108170062053440751665152000000000
41:42:NaN

42:-1:NaN
42:0:1
42:1:42
42:2:1722
42:3:68880
42:4:2686320
42:5:102080160
42:6:3776965920
42:7:135970773120
42:8:4758977059200
42:9:161805220012800
42:10:5339572260422400
42:11:170866312333516800
42:12:5296855682339020800
42:13:158905670470170624000
42:14:4608264443634948096000
42:15:129031404421778546688000
42:16:3483847919388020760576000
42:17:90580045904088539774976000
42:18:2264501147602213494374400000
42:19:54348027542453123864985600000
42:20:1250004633476421848894668800000
42:21:27500101936481280675682713600000
42:22:577502140666106894189336985600000
42:23:11550042813322137883786739712000000
42:24:219450813453120619791948054528000000
42:25:3950114642156171156255064981504000000
42:26:67151948916654909656336104685568000000
42:27:1074431182666478554501377674969088000000
42:28:16116467739997178317520665124536320000000
42:29:225630548359960496445289311743508480000000
42:30:2933197128679486453788761052665610240000000
42:31:35198365544153837445465132631987322880000000
42:32:387182020985692211900116458951860551680000000
42:33:3871820209856922119001164589518605516800000000
42:34:34846381888712299071010481305667449651200000000
42:35:278771055109698392568083850445339597209600000000
42:36:1951397385767888747976586953117377180467200000000
42:37:11708384314607332487859521718704263082803200000000
42:38:58541921573036662439297608593521315414016000000000
42:39:234167686292146649757190434374085261656064000000000
42:40:702503058876439949271571303122255784968192000000000
42:41:1405006117752879898543142606244511569936384000000000
42:42:1405006117752879898543142606244511569936384000000000
42:43:NaN

43:-1:NaN
43:0:1
43:1:43
43:2:1806
43:3:74046
43:4:2961840
43:5:115511760
43:6:4389446880
43:7:162409534560
43:8:5846743244160
43:9:204636013545600
43:10:6957624460550400
43:11:229601607198163200
43:12:7347251430341222400
43:13:227764794340577894400
43:14:6832943830217336832000
43:15:198155371076302768128000
43:16:5548350390136477507584000
43:17:149805460533684892704768000
43:18:3894941973875807210323968000
43:19:97373549346895180258099200000
43:20:2336965184325484326194380800000
43:21:53750199239486139502470758400000
43:22:1182504383268695069054356684800000
43:23:24832592048642596450141490380800000
43:24:496651840972851929002829807616000000
43:25:9436384978484186651053766344704000000
43:26:169854929612715359718967794204672000000
43:27:2887533803416161115222452501479424000000
43:28:46200540854658577843559240023670784000000
43:29:693008112819878667653388600355061760000000
43:30:9702113579478301347147440404970864640000000
43:31:126127476533217917512916725264621240320000000
43:32:1513529718398615010155000703175454883840000000
43:33:16648826902384765111705007734930003722240000000
43:34:166488269023847651117050077349300037222400000000
43:35:1498394421214628860053450696143700335001600000000
43:36:11987155369717030880427605569149602680012800000000
43:37:83910087588019216162993238984047218760089600000000
43:38:503460525528115296977959433904283312560537600000000
43:39:2517302627640576484889797169521416562802688000000000
43:40:10069210510562305939559188678085666251210752000000000
43:41:30207631531686917818677566034256998753632256000000000
43:42:60415263063373835637355132068513997507264512000000000
43:43:60415263063373835637355132068513997507264512000000000
43:44:NaN

44:-1:NaN
44:0:1
44:1:44
44:2:1892
44:3:79464
44:4:3258024
44:5:130320960
44:6:5082517440
44:7:193135662720
44:8:7146019520640
44:9:257256702743040
44:10:9003984596006400
44:11:306135476264217600
44:12:10102470716719180800
44:13:323279062935013785600
44:14:10021650950985427353600
44:15:300649528529562820608000
44:16:8718836327357321797632000
44:17:244127417166005010333696000
44:18:6591440263482135279009792000
44:19:171377446850535517254254592000
44:20:4284436171263387931356364800000
44:21:102826468110321310352552755200000
44:22:2365008766537390138108713369600000
44:23:52030192863822583038391694131200000
44:24:1092634050140274243806225576755200000
44:25:21852681002805484876124511535104000000
44:26:415200939053304212646365719166976000000
44:27:7473616902959475827634582945005568000000
44:28:127051487350311089069787910065094656000000
44:29:2032823797604977425116606561041514496000000
44:30:30492356964074661376749098415622717440000000
44:31:426892997497045259274487377818718044160000000
44:32:5549608967461588370568335911643334574080000000
44:33:66595307609539060446820030939720014888960000000
44:34:732548383704929664915020340336920163778560000000
44:35:7325483837049296649150203403369201637785600000000
44:36:65929354533443669842351830630322814740070400000000
44:37:527434836267549358738814645042582517920563200000000
44:38:3692043853872845511171702515298077625443942400000000
44:39:22152263123237073067030215091788465752663654400000000
44:40:110761315616185365335151075458942328763318272000000000
44:41:443045262464741461340604301835769315053273088000000000
44:42:1329135787394224384021812905507307945159819264000000000
44:43:2658271574788448768043625811014615890319638528000000000
44:44:2658271574788448768043625811014615890319638528000000000
44:45:NaN

45:-1:NaN
45:0:1
45:1:45
45:2:1980
45:3:85140
45:4:3575880
45:5:146611080
45:6:5864443200
45:7:228713284800
45:8:8691104822400
45:9:321570878428800
45:10:11576551623436800
45:11:405179306820288000
45:12:13776096431889792000
45:13:454611182252363136000
45:14:14547557832075620352000
45:15:450974292794344230912000
45:16:13529228783830326927360000
45:17:392347634731079480893440000
45:18:10985733772470225465016320000
45:19:296614811856696087555440640000
45:20:7711985108274098276441456640000
45:21:192799627706852456911036416000000
45:22:4627191064964458965864873984000000
45:23:106425394494182556214892101632000000
45:24:2341358678872016236727626235904000000
45:25:49168532256312340971280150953984000000
45:26:983370645126246819425603019079680000000
45:27:18684042257398689569086457362513920000000
45:28:336312760633176412243556232525250560000000
45:29:5717316930763999008140455952929259520000000
45:30:91477070892223984130247295246868152320000000
45:31:1372156063383359761953709428703022284800000000
45:32:19210184887367036667351932001842311987200000000
45:33:249732403535771476675575116023950055833600000000
45:34:2996788842429257720106901392287400670003200000000
45:35:32964677266721834921175915315161407370035200000000
45:36:329646772667218349211759153151614073700352000000000
45:37:2966820954004965142905832378364526663303168000000000
45:38:23734567632039721143246659026916213306425344000000000
45:39:166141973424278048002726613188413493144977408000000000
45:40:996851840545668288016359679130480958869864448000000000
45:41:4984259202728341440081798395652404794349322240000000000
45:42:19937036810913365760327193582609619177397288960000000000
45:43:59811110432740097280981580747828857532191866880000000000
45:44:119622220865480194561963161495657715064383733760000000000
45:45:119622220865480194561963161495657715064383733760000000000
45:46:NaN

46:-1:NaN
46:0:1
46:1:46
46:2:2070
46:3:91080
46:4:3916440
46:5:164490480
46:6:6744109680
46:7:269764387200
46:8:10520811100800
46:9:399790821830400
46:10:14792260407724800
46:11:532521374678092800
46:12:18638248113733248000
46:13:633700435866930432000
46:14:20912114383608704256000
46:15:669187660275478536192000
46:16:20744817468539834621952000
46:17:622344524056195038658560000
46:18:18047991197629656121098240000
46:19:505343753533630371390750720000
46:20:13644281345408020027550269440000
46:21:354751314980608520716307005440000
46:22:8868782874515213017907675136000000
46:23:212850788988365112429784203264000000
46:24:4895568146732397585885036675072000000
46:25:107702499228112746889470806851584000000
46:26:2261752483790367684678886943883264000000
46:27:45235049675807353693577738877665280000000
46:28:859465943840339720177977038675640320000000
46:29:15470386989126114963203586696161525760000000
46:30:262996578815143954374460973834745937920000000
46:31:4207945261042303269991375581355935006720000000
46:32:63119178915634549049870633720339025100800000000
46:33:883668504818883686698188872084746351411200000000
46:34:11487690562645487927076455337101702568345600000000
46:35:137852286751745855124917464045220430820147200000000
46:36:1516375154269204406374092104497424739021619200000000
46:37:15163751542692044063740921044974247390216192000000000
46:38:136473763884228396573668289404768226511945728000000000
46:39:1091790111073827172589346315238145812095565824000000000
46:40:7642530777516790208125424206667020684668960768000000000
46:41:45855184665100741248752545240002124108013764608000000000
46:42:229275923325503706243762726200010620540068823040000000000
46:43:917103693302014824975050904800042482160275292160000000000
46:44:2751311079906044474925152714400127446480825876480000000000
46:45:5502622159812088949850305428800254892961651752960000000000
46:46:5502622159812088949850305428800254892961651752960000000000
46:47:NaN

47:-1:NaN
47:0:1
47:1:47
47:2:2162
47:3:97290
47:4:4280760
47:5:184072680
47:6:7731052560
47:7:316973154960
47:8:12678926198400
47:9:494478121737600
47:10:18790168626028800
47:11:695236239163065600
47:12:25028504609870361600
47:13:875997661345462656000
47:14:29783920485745730304000
47:15:982869376029609100032000
47:16:31451820032947491201024000
47:17:975006421021372227231744000
47:18:29250192630641166816952320000
47:19:848255586288593837691617280000
47:20:23751156416080627455365283840000
47:21:641281223234176941294862663680000
47:22:16673311804088600473666429255680000
47:23:416832795102215011841660731392000000
47:24:10003987082453160284199857553408000000
47:25:230091702896422686536596723728384000000
47:26:5062017463721299103805127922024448000000
47:27:106302366738147281179907686362513408000000
47:28:2126047334762945623598153727250268160000000
47:29:40394899360495966848364920817755095040000000
47:30:727108188488927403270568574719591710720000000
47:31:12360839204311765855599665770233059082240000000
47:32:197773427268988253689594652323728945315840000000
47:33:2966601409034823805343919784855934179737600000000
47:34:41532419726487533274814876987983078516326400000000
47:35:539921456444337932572593400843780020712243200000000
47:36:6479057477332055190871120810125360248546918400000000
47:37:71269632250652607099582328911378962734016102400000000
47:38:712696322506526070995823289113789627340161024000000000
47:39:6414266902558734638962409602024106646061449216000000000
47:40:51314135220469877111699276816192853168491593728000000000
47:41:359198946543289139781894937713349972179441156096000000000
47:42:2155193679259734838691369626280099833076646936576000000000
47:43:10775968396298674193456848131400499165383234682880000000000
47:44:43103873585194696773827392525601996661532938731520000000000
47:45:129311620755584090321482177576805989984598816194560000000000
47:46:258623241511168180642964355153611979969197632389120000000000
47:47:258623241511168180642964355153611979969197632389120000000000
47:48:NaN

48:-1:NaN
48:0:1
48:1:48
48:2:2256
48:3:103776
48:4:4669920
48:5:205476480
48:6:8835488640
48:7:371090522880
48:8:15214711438080
48:9:608588457523200
48:10:23734949843404800
48:11:901928094049382400
48:12:33371339479827148800
48:13:1201368221273777356800
48:14:42047887744582207488000
48:15:1429628183315795054592000
48:16:47177730049421236801536000
48:17:1509687361581479577649152000
48:18:46800308209025866907123712000
48:19:1404009246270776007213711360000
48:20:40716268141852504209197629440000
48:21:1140055507971870117857533624320000
48:22:30781498715240493182153407856640000
48:23:800318966596252822735988604272640000
48:24:20007974164906320568399715106816000000
48:25:480191379957751693641593162563584000000
48:26:11044401739028288953756642738962432000000
48:27:242976838258622356982646140257173504000000
48:28:5102513603431069496635568945400643584000000
48:29:102050272068621389932711378908012871680000000
48:30:1938955169303806408721516199252244561920000000
48:31:34901193047468515356987291586540402114560000000
48:32:593320281806964761068783956971186835947520000000
48:33:9493124508911436177100543311538989375160320000000
48:34:142396867633671542656508149673084840627404800000000
48:35:1993556146871401597191114095423187768783667200000000
48:36:25916229909328220763484483240501440994187673600000000
48:37:310994758911938649161813798886017291930252083200000000
48:38:3420942348031325140779951787746190211232772915200000000
48:39:34209423480313251407799517877461902112327729152000000000
48:40:307884811322819262670195660897157119010949562368000000000
48:41:2463078490582554101361565287177256952087596498944000000000
48:42:17241549434077878709530957010240798664613175492608000000000
48:43:103449296604467272257185742061444791987679052955648000000000
48:44:517246483022336361285928710307223959938395264778240000000000
48:45:2068985932089345445143714841228895839753581059112960000000000
48:46:6206957796268036335431144523686687519260743177338880000000000
48:47:12413915592536072670862289047373375038521486354677760000000000
48:48:12413915592536072670862289047373375038521486354677760000000000
48:49:NaN

49:-1:NaN
49:0:1
49:1:49
49:2:2352
49:3:110544
49:4:5085024
49:5:228826080
49:6:10068347520
49:7:432938943360
49:8:18183435621120
49:9:745520860465920
49:10:29820834418636800
49:11:1163012542326835200
49:12:44194476608419737600
49:13:1635195634511530291200
49:14:58867042842415090483200
49:15:2060346499484528166912000
49:16:70051780982473957675008000
49:17:2311708772421640603275264000
49:18:73974680717492499304808448000
49:19:2293215102242267478449061888000
49:20:68796453067268024353471856640000
49:21:1995097138950772706250683842560000
49:22:55862719890621635775019147591680000
49:23:1508293437046784165925516984975360000
49:24:39215629363216388314063441609359360000
49:25:980390734080409707851586040233984000000
49:26:23529377617929832988438064965615616000000
49:27:541175685212386158734075494209159168000000
49:28:11905865074672495492149660872601501696000000
49:29:250023166568122405335142878324631535616000000
49:30:5000463331362448106702857566492630712320000000
49:31:95008803295886514027354293763359983534080000000
49:32:1710158459325957252492377287740479703613440000000
49:33:29072693808541273292370413891588154961428480000000
49:34:465163100936660372677926622265410479382855680000000
49:35:6977446514049905590168899333981157190742835200000000
49:36:97684251196698678262364590675736200670399692800000000
49:37:1269895265557082817410739678784570608715196006400000000
49:38:15238743186684993808928876145414847304582352076800000000
49:39:167626175053534931898217637599563320350405872844800000000
49:40:1676261750535349318982176375995633203504058728448000000000
49:41:15086355754818143870839587383960698831536528556032000000000
49:42:120690846038545150966716699071685590652292228448256000000000
49:43:844835922269816056767016893501799134566045599137792000000000
49:44:5069015533618896340602101361010794807396273594826752000000000
49:45:25345077668094481703010506805053974036981367974133760000000000
49:46:101380310672377926812042027220215896147925471896535040000000000
49:47:304140932017133780436126081660647688443776415689605120000000000
49:48:608281864034267560872252163321295376887552831379210240000000000
49:49:608281864034267560872252163321295376887552831379210240000000000
49:50:NaN

50:-1:NaN
50:0:1
50:1:50
50:2:2450
50:3:117600
50:4:5527200
50:5:254251200
50:6:11441304000
50:7:503417376000
50:8:21646947168000
50:9:909171781056000
50:10:37276043023296000
50:11:1491041720931840000
50:12:58150627116341760000
50:13:2209723830420986880000
50:14:81759781725576514560000
50:15:2943352142120754524160000
50:16:103017324974226408345600000
50:17:3502589049123697883750400000
50:18:115585438621082030163763200000
50:19:3698734035874624965240422400000
50:20:114660755112113373922453094400000
50:21:3439822653363401217673592832000000
50:22:99754856947538635312534192128000000
50:23:2793135994531081788750957379584000000
50:24:75414671852339208296275849248768000000
50:25:1960781468160819415703172080467968000000
50:26:49019536704020485392579302011699200000000
50:27:1176468880896491649421903248280780800000000
50:28:27058784260619307936703774710457958400000000
50:29:595293253733624774607483043630075084800000000
50:30:12501158328406120266757143916231576780800000000
50:31:250023166568122405335142878324631535616000000000
50:32:4750440164794325701367714688167999176704000000000
50:33:85507922966297862624618864387023985180672000000000
50:34:1453634690427063664618520694579407748071424000000000
50:35:23258155046833018633896331113270523969142784000000000
50:36:348872325702495279508444966699057859537141760000000000
50:37:4884212559834933913118229533786810033519984640000000000
50:38:63494763277854140870536983939228530435759800320000000000
50:39:761937159334249690446443807270742365229117603840000000000
50:40:8381308752676746594910881879978166017520293642240000000000
50:41:83813087526767465949108818799781660175202936422400000000000
50:42:754317787740907193541979369198034941576826427801600000000000
50:43:6034542301927257548335834953584279532614611422412800000000000
50:44:42241796113490802838350844675089956728302279956889600000000000
50:45:253450776680944817030105068050539740369813679741337600000000000
50:46:1267253883404724085150525340252698701849068398706688000000000000
50:47:5069015533618896340602101361010794807396273594826752000000000000
50:48:15207046600856689021806304083032384422188820784480256000000000000
50:49:30414093201713378043612608166064768844377641568960512000000000000
50:50:30414093201713378043612608166064768844377641568960512000000000000
50:51:NaN

# big numbers

340282366920938463463374607431768211456:0:1
340282366920938463463374607431768211456:1:340282366920938463463374607431768211456
340282366920938463463374607431768211456:2:115792089237316195423570985008687907852929702298719625575994209400481361428480
